#include "font.h"

// ----------------------------------------------------------------------------------------------------------------------------

CFont::CFont()
{
	SetDefaults();
}

CFont::~CFont()
{
}

bool CFont::Init(char *FileName)
{
	if(!GLEW_ARB_texture_rectangle)
	{
		ErrorLog.Append("GL_ARB_texture_rectangle not supported!");
		return false;
	}

	CString DirectoryFileName = ModuleDirectory + "Textures\\" + FileName;

	FIBITMAP *dib = GetBitmap(DirectoryFileName);

	if(dib == NULL)
	{
		ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
		return false;
	}

	int BPP = FreeImage_GetBPP(dib);
	int Pitch = FreeImage_GetPitch(dib);

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileName);
		FreeImage_Unload(dib);
		return false;
	}
	
	Destroy();

	BYTE *Data = FreeImage_GetBits(dib);
	BYTE *NewData = new BYTE[Width * Height * 4];

	BPP /= 8;

	for(int y = 0; y < Height; y++)
	{
		for(int x = 0; x < Width; x++)
		{
			int Pos1 = y * Pitch + x * BPP;
			int Pos2 = (y * Width + x) * 4;

			for(int i = 0; i < 3; i++)
			{
				NewData[Pos2 + i] = Data[Pos1 + i];
			}

			NewData[Pos2 + 3] = 0;
		}
	}

	for(int y = 0; y < Height; y++)
	{
		for(int x = 0; x < Width; x++)
		{
			int Pos = (y * Width + x) * 4;

			BYTE Alpha = 0;

			if(NewData[Pos + 0] || NewData[Pos + 1] || NewData[Pos + 2])
			{
				Alpha = 255;
			}
			else
			{
				for(int yy = -1; yy <= 1; yy++)
				{
					int yyy = y + yy;

					if(yyy >= 0 && yyy < Height)
					{
						for(int xx = -1; xx <= 1; xx++)
						{
							int xxx = x + xx;

							if(xxx >= 0 && xxx < Width)
							{
								int Pos = (yyy * Width + xxx) * 4;

								if(NewData[Pos + 0] || NewData[Pos + 1] || NewData[Pos + 2])
								{
									Alpha = 192;
								}
							}
						}
					}
				}
			}

			NewData[Pos + 3] = Alpha;
		}
	}

	LetterWidth = new float[200];

	for(int Y = 0; Y < 10; Y++)
	{
		for(int X = 0; X < 20; X++)
		{
			int x = X * 18, y = Y * 24, minx = x + 18, maxx = x;

			for(int yy = y; yy < y + 24; yy++)
			{
				for(int xx = x; xx < x + 18; xx++)
				{
					int Pos = (yy * Width + xx) * 4;

					if(NewData[Pos + 3])
					{
						if(xx < minx) minx = xx;
						if(xx > maxx) maxx = xx;
					}
				}
			}

			float *letterwidth = LetterWidth + (9 - Y) * 20 + X;
			*letterwidth = (float)(maxx - minx + 2);
			if(*letterwidth <= 0.0f) *letterwidth = 6.0f;

			int offset = minx - x;

			for(int yy = y; yy < y + 24; yy++)
			{
				for(int xx = x; xx < x + 18 - offset; xx++)
				{
					int Pos1 = (yy * Width + xx) * 4;
					int Pos2 = (yy * Width + xx + offset) * 4;

					for(int i = 0; i < 4; i++)
					{
						NewData[Pos1 + i] = NewData[Pos2 + i];
					}
				}

				for(int xx = x + 18 - offset; xx < x + 18; xx++)
				{
					int Pos = (yy * Width + xx) * 4;

					for(int i = 0; i < 4; i++)
					{
						NewData[Pos + i] = 0;
					}
				}
			}
		}
	}

	glGenTextures(1, &Texture);
	glBindTexture(GL_TEXTURE_RECTANGLE_ARB, Texture);
	glTexParameteri(GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_RECTANGLE_ARB, 0, GL_RGBA8, Width, Height, 0, GL_BGRA, GL_UNSIGNED_BYTE, NewData);
	glBindTexture(GL_TEXTURE_RECTANGLE_ARB, 0);

	delete [] NewData;

	FreeImage_Unload(dib);

	return true;
}

void CFont::Printf(char *Text, int X, int Y)
{
	float Start = (float)X + 3.0f, x = Start, y = (float)Y, xi = 18.0f, yi = 24.0f;

	size_t Length = strlen(Text);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glColor3f(1.0f, 1.0f, 1.0f);
	glEnable(GL_TEXTURE_RECTANGLE_ARB);
	glBindTexture(GL_TEXTURE_RECTANGLE_ARB, Texture);
	glBegin(GL_QUADS);

	for(size_t i = 0; i < Length; i++)
	{
		if(Text[i] == '\n')
		{
			x = Start;
			y += yi;
		}
		else
		{
			int p = Text[i] - 33;

			if(p < 0) p += 200;
			if(p > 199) p -= 200;

			float s = (float)(p % 20) * xi;
			float t = (float)(9 - p / 20) * yi;

			float xi = LetterWidth[p];

			glTexCoord2f(s, t + yi); glVertex2f(x, y);
			glTexCoord2f(s + xi, t + yi); glVertex2f(x + xi, y);
			glTexCoord2f(s + xi, t); glVertex2f(x + xi, y + yi);
			glTexCoord2f(s, t); glVertex2f(x, y + yi);

			x += xi;
		}
	}

	glEnd();
	glBindTexture(GL_TEXTURE_RECTANGLE_ARB, 0);
	glDisable(GL_TEXTURE_RECTANGLE_ARB);
	glDisable(GL_BLEND);
}

void CFont::Destroy()
{
	glDeleteTextures(1, &Texture);
	
	delete [] LetterWidth;
	
	SetDefaults();
}

FIBITMAP *CFont::GetBitmap(char *FileName)
{
	FREE_IMAGE_FORMAT fif = FreeImage_GetFileType(FileName);

	if(fif == FIF_UNKNOWN)
	{
		fif = FreeImage_GetFIFFromFilename(FileName);
	}

	if(fif == FIF_UNKNOWN)
	{
		return NULL;
	}

	FIBITMAP *dib = NULL;

	if(FreeImage_FIFSupportsReading(fif))
	{
		dib = FreeImage_Load(fif, FileName);
	}

	if(dib != NULL)
	{
		Width = FreeImage_GetWidth(dib);
		Height = FreeImage_GetHeight(dib);

		if(Width == 0 || Height == 0)
		{
			FreeImage_Unload(dib);
			return NULL;
		}
	}

	return dib;
}

void CFont::SetDefaults()
{
	Texture = 0;
	LetterWidth = NULL;
}
