#include "openglview.h"

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView *OpenGLView = NULL;

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK OpenGLViewWndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uiMsg)
	{
		case WM_COMMAND:
			if(OpenGLView != NULL) OpenGLView->OnCommand(LOWORD(wParam));
			break;

		case WM_KEYDOWN:
			if(OpenGLView != NULL) OpenGLView->OnKeyDown((UINT)wParam);
			break;

		case WM_KILLFOCUS:
			if(OpenGLView != NULL) OpenGLView->OnKillFocus();
			break;

		case WM_LBUTTONDOWN:
			if(OpenGLView != NULL) OpenGLView->OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_MOUSEMOVE:
			if(OpenGLView != NULL) OpenGLView->OnMouseMove(LOWORD(lParam), HIWORD(lParam));
			break;

		case 0x020A: // WM_MOUSEWHEEL
			if(OpenGLView != NULL) OpenGLView->OnMouseWheel(HIWORD(wParam));
			break;

		case WM_PAINT:
			if(OpenGLView != NULL) OpenGLView->OnPaint();
			break;

		case WM_RBUTTONDOWN:
			if(OpenGLView != NULL) OpenGLView->OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_RBUTTONUP:
			if(OpenGLView != NULL) OpenGLView->OnRButtonUp(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_SETFOCUS:
			if(OpenGLView != NULL) OpenGLView->OnSetFocus();
			break;

		case WM_SIZE:
			if(OpenGLView != NULL) OpenGLView->OnSize(LOWORD(lParam), HIWORD(lParam));
			break;

		default:
			return DefWindowProc(hWnd, uiMsg, wParam, lParam);
	}

	return 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView::COpenGLView()
{
	MyOpenGLRenderer = new CMyOpenGLRenderer();

	OpenGLView = this;

	Focus = false;
}

COpenGLView::~COpenGLView()
{
	delete MyOpenGLRenderer;
}

bool COpenGLView::Init(HINSTANCE hInstance, HWND hWndParent, int Width, int Height, int Samples)
{
	PopupMenu = LoadMenu(hInstance, MAKEINTRESOURCE(IDR_POPUP_MENU));

	this->Width = Width;
	this->Height = Height;

	WNDCLASSEX WndClassEx;

	memset(&WndClassEx, 0, sizeof(WNDCLASSEX));

	WndClassEx.cbSize = sizeof(WNDCLASSEX);
	WndClassEx.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	WndClassEx.lpfnWndProc = OpenGLViewWndProc;
	WndClassEx.hInstance = hInstance;
	WndClassEx.hCursor = LoadCursor(NULL, IDC_ARROW);
	WndClassEx.lpszClassName = "OpenGLViewClass";

	if(RegisterClassEx(&WndClassEx) == 0)
	{
		ErrorLog.Set("RegisterClassEx failed!");
		return false;
	}

	DWORD Style = WS_CHILD | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	hWnd = CreateWindow(WndClassEx.lpszClassName, NULL, Style, 0, 0, Width, Height, hWndParent, NULL, hInstance, NULL);

	if(hWnd == NULL)
	{
		ErrorLog.Set("CreateWindowEx failed!");
		return false;
	}

	HDC hDC = GetDC(hWnd);

	if(hDC == NULL)
	{
		ErrorLog.Set("GetDC failed!");
		return false;
	}

	PIXELFORMATDESCRIPTOR pfd;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DOUBLEBUFFER | PFD_SUPPORT_OPENGL | PFD_DRAW_TO_WINDOW;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 32;
	pfd.cDepthBits = 24;
	pfd.iLayerType = PFD_MAIN_PLANE;
 
	int PixelFormat = ChoosePixelFormat(hDC, &pfd);

	if(PixelFormat == 0)
	{
		ErrorLog.Set("ChoosePixelFormat failed!");
		return false;
	}

	static int MSAAPixelFormat = 0;

	if(SetPixelFormat(hDC, MSAAPixelFormat == 0 ? PixelFormat : MSAAPixelFormat, &pfd) == FALSE)
	{
		ErrorLog.Set("SetPixelFormat failed!");
		return false;
	}

	hGLRC = wglCreateContext(hDC);

	if(hGLRC == NULL)
	{
		ErrorLog.Set("wglCreateContext failed!");
		return false;
	}

	if(wglMakeCurrent(hDC, hGLRC) == FALSE)
	{
		ErrorLog.Set("wglMakeCurrent (1) failed!");
		return false;
	}

	if(glewInit() != GLEW_OK)
	{
		ErrorLog.Set("glewInit failed!");
		return false;
	}

	if(MSAAPixelFormat == 0 && Samples > 0)
	{
		if(WGLEW_ARB_pixel_format && GLEW_ARB_multisample)
		{
			while(Samples > 0)
			{
				UINT NumFormats = 0;

				int PFAttribs[] =
				{
					WGL_DRAW_TO_WINDOW_ARB, GL_TRUE,
					WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
					WGL_DOUBLE_BUFFER_ARB, GL_TRUE,
					WGL_PIXEL_TYPE_ARB, WGL_TYPE_RGBA_ARB,
					WGL_COLOR_BITS_ARB, 32,
					WGL_DEPTH_BITS_ARB, 24,
					WGL_ACCELERATION_ARB, WGL_FULL_ACCELERATION_ARB,
					WGL_SAMPLE_BUFFERS_ARB, GL_TRUE,
					WGL_SAMPLES_ARB, Samples,
					0
				};

				if(wglChoosePixelFormatARB(hDC, PFAttribs, NULL, 1, &MSAAPixelFormat, &NumFormats) == TRUE && NumFormats > 0) break;

				Samples--;
			}

			wglDeleteContext(hGLRC);
			DestroyWindow(hWnd);
			UnregisterClass(WndClassEx.lpszClassName, hInstance);

			return Init(hInstance, hWndParent, Width, Height, Samples);
		}
		else
		{
			Samples = 0;
		}
	}

	this->Samples = Samples;

	/*if(WGLEW_ARB_create_context)
	{
		wglDeleteContext(hGLRC);

		int GLRCAttribs[] =
		{
			WGL_CONTEXT_MAJOR_VERSION_ARB, 3,
			WGL_CONTEXT_MINOR_VERSION_ARB, 3,
			WGL_CONTEXT_FLAGS_ARB, WGL_CONTEXT_FORWARD_COMPATIBLE_BIT_ARB,
			0
		};

		hGLRC = wglCreateContextAttribsARB(hDC, 0, GLRCAttribs);

		if(hGLRC == NULL)
		{
			ErrorLog.Set("wglCreateContextAttribsARB failed!");
			return false;
		}

		if(wglMakeCurrent(hDC, hGLRC) == FALSE)
		{
			ErrorLog.Set("wglMakeCurrent (2) failed!");
			return false;
		}
	}
	else
	{
		ErrorLog.Set("WGL_ARB_create_context not supported!");
		return false;
	}*/

	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &gl_max_texture_size);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &gl_max_texture_max_anisotropy_ext);
	}

	if(WGLEW_EXT_swap_control)
	{
		wglSwapIntervalEXT(0);
	}

	if(MyOpenGLRenderer != NULL)
	{
		return MyOpenGLRenderer->Init();
	}

	return true;
}

void COpenGLView::Move(int X, int Y, int Width, int Height)
{
	MoveWindow(hWnd, X, Y, Width, Height, FALSE);
}

void COpenGLView::SetFocus()
{
	if(!Focus)
	{
		::SetFocus(hWnd);
	}
}

void COpenGLView::Show()
{
	ShowWindow(hWnd, SW_SHOWNORMAL);
}

void COpenGLView::Destroy()
{
	if(MyOpenGLRenderer != NULL)
	{
		MyOpenGLRenderer->Destroy();
	}

	wglDeleteContext(hGLRC);
	DestroyWindow(hWnd);
	DestroyMenu(PopupMenu);
}

void COpenGLView::OnCommand(int ID)
{
	if(MyOpenGLRenderer != NULL)
	{
		MyOpenGLRenderer->OnCommand(ID);
	}
}

void COpenGLView::OnKeyDown(UINT Key)
{
	if(MyOpenGLRenderer != NULL)
	{
		MyOpenGLRenderer->OnKeyDown(Key);
	}
}

void COpenGLView::OnKillFocus()
{
	Focus = false;
}

void COpenGLView::OnLButtonDown(int X, int Y)
{
	SetFocus();

	MouseMoved = false;

	if(MyOpenGLRenderer != NULL)
	{
		MyOpenGLRenderer->OnLButtonDown(X, Y);
	}
}

void COpenGLView::OnMouseMove(int X, int Y)
{
	MouseMoved = true;

	if(MyOpenGLRenderer != NULL)
	{
		MyOpenGLRenderer->OnMouseMove(X, Y, GetKeyState(VK_LBUTTON) & 0x80 ? true : false, GetKeyState(VK_RBUTTON) & 0x80 ? true : false);
	}
}

void COpenGLView::OnMouseWheel(short zDelta)
{
	if(MyOpenGLRenderer != NULL)
	{
		MyOpenGLRenderer->OnMouseWheel(zDelta);
	}
}

void COpenGLView::OnPaint()
{
	static DWORD LastFrameTime = GetTickCount(), LastFPSTime = LastFrameTime, FPS = 0;

	PAINTSTRUCT ps;

	HDC hDC = BeginPaint(hWnd, &ps);

	DWORD Time = GetTickCount();

	float FrameTime = float(Time - LastFrameTime) * 0.001f;
	LastFrameTime = Time;

	if(Time - LastFPSTime > 1000)
	{
		if(MyOpenGLRenderer != NULL)
		{
			MyOpenGLRenderer->Text.Set("%dx%d, ", Width, Height);
			MyOpenGLRenderer->Text.Append("ATF %dx, ", gl_max_texture_max_anisotropy_ext);
			MyOpenGLRenderer->Text.Append("MSAA %dx\n", Samples);
			MyOpenGLRenderer->Text.Append("FPS: %d\n", FPS);
			MyOpenGLRenderer->Text.Append("%s\n", glGetString(GL_RENDERER));
		}

		LastFPSTime = Time;
		FPS = 0;
	}
	else
	{
		FPS++;
	}

	if(MyOpenGLRenderer != NULL)
	{
		bool Animated = MyOpenGLRenderer->Animate(FrameTime);

		SHORT Keys = 0x0000;

		if(GetKeyState('W') & 0x80) Keys |= CAMERA_KEY_W;
		if(GetKeyState('S') & 0x80) Keys |= CAMERA_KEY_S;
		if(GetKeyState('A') & 0x80) Keys |= CAMERA_KEY_A;
		if(GetKeyState('D') & 0x80) Keys |= CAMERA_KEY_D;
		if(GetKeyState('R') & 0x80) Keys |= CAMERA_KEY_R;
		if(GetKeyState('F') & 0x80) Keys |= CAMERA_KEY_F;
		if(GetKeyState('Q') & 0x80) Keys |= CAMERA_KEY_Q;
		if(GetKeyState('E') & 0x80) Keys |= CAMERA_KEY_E;
		if(GetKeyState('C') & 0x80) Keys |= CAMERA_KEY_C;
		if(GetKeyState(VK_SPACE) & 0x80) Keys |= CAMERA_KEY_SPACE;
		if(GetKeyState(VK_SHIFT) & 0x80) Keys |= CAMERA_KEY_SHIFT;
		if(GetKeyState(VK_CONTROL) & 0x80) Keys |= CAMERA_KEY_CONTROL;

		vec3 Movement;

		if(MyOpenGLRenderer->OnCameraKeys(Keys, FrameTime, Movement))
		{
			MyOpenGLRenderer->MoveCamera(Movement);
		}

		MyOpenGLRenderer->Render();
	}

	SwapBuffers(hDC);

	EndPaint(hWnd, &ps);

	if(Focus)
	{
		InvalidateRect(hWnd, NULL, FALSE);
	}
}

void COpenGLView::OnRButtonDown(int X, int Y)
{
	SetFocus();

	MouseMoved = false;

	if(MyOpenGLRenderer != NULL)
	{
		MyOpenGLRenderer->OnRButtonDown(X, Y);
	}
}

void COpenGLView::OnRButtonUp(int X, int Y)
{
	if(!MouseMoved)
	{
		POINT Point;

		Point.x = X;
		Point.y = Y;

		ClientToScreen(hWnd, &Point);

		TrackPopupMenu(GetSubMenu(PopupMenu, 0), TPM_LEFTALIGN, Point.x, Point.y, 0, hWnd, NULL);
	}
}

void COpenGLView::OnSetFocus()
{
	Focus = true;

	InvalidateRect(hWnd, NULL, FALSE);
}

void COpenGLView::OnSize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	if(MyOpenGLRenderer != NULL)
	{
		MyOpenGLRenderer->Resize(Width, Height);
	}
}
