#include <windows.h>

#include "string.h"
#include "glmath.h"

// ----------------------------------------------------------------------------------------------------------------------------

class CCamera
{
public:
	vec3 X, Y, Z, Reference, Position;
	mat4x4 Vin, Pin, VPin, RayMatrix;

public:
	CCamera();
	~CCamera();

	void CalculateRayMatrix();
	void Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference = false);
	bool OnKeyDown(UINT nChar);
	void OnMouseMove(int dx, int dy);
	void OnMouseWheel(short zDelta);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CTriangle
{
public:
	float D, D1, D2, D3, lab, lbc, lca;
	vec3 a, b, c, ab, bc, ca, Color, N, N1, N2, N3;

public:
	CTriangle();
	CTriangle(const vec3 &a, const vec3 &b, const vec3 &c, const vec3 &Color);

	bool Inside(float x, float y, float z);
	bool Inside(const vec3 &Point);
	bool Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance, float &Distance, vec3 &Point);
	bool Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance, float &Distance);
	bool Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance);
};

// ----------------------------------------------------------------------------------------------------------------------------

class RTData
{
public:
	float Distance, TestDistance;
	vec3 Color, Point, TestPoint;
	CTriangle *Triangle;

public:
	RTData();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CVoxel
{
public:
	CTriangle **Triangles;
	int TrianglesCount;

protected:
	int MaxTrianglesCount;
	float Size;
	vec3 Min, Max, MinE, MaxE;

public:
	CVoxel();
	~CVoxel();

	void Add(CTriangle *Triangle);
	void Delete();
	bool Inside(const vec3 &Point);
	bool Intersect(CTriangle *Triangle);
	bool IntersectEdgesX(CTriangle *Triangle, float x, float y1, float y2, float z1, float z2);
	bool IntersectEdgesY(CTriangle *Triangle, float y, float x1, float x2, float z1, float z2);
	bool IntersectEdgesZ(CTriangle *Triangle, float z, float x1, float x2, float y1, float y2);
	bool IntersectFacesX(CTriangle *Triangle, float D1, float D2);
	bool IntersectFacesY(CTriangle *Triangle, float D1, float D2);
	bool IntersectFacesZ(CTriangle *Triangle, float D1, float D2);
	void Set(const vec3 &Min, float Size);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CUniformGrid
{
protected:
	vec3 Min, Max;

protected:
	int X, Y, Z, Xm1, Ym1, Zm1, XY, XYZ;
	float VoxelSize;
	CVoxel *Voxels;

public:
	CUniformGrid();
	~CUniformGrid();

public:
	void Delete();
	void Generate(CTriangle *Triangles, int TrianglesCount, float VoxelSize = 1.0f);
	vec3 Traverse(const vec3 &Voxel, const vec3 &Origin, const vec3 &Ray);

	float VoxelToWorldX(float x);
	float VoxelToWorldY(float y);
	float VoxelToWorldZ(float z);
	vec3 VoxelToWorld(const vec3 &Voxel);
	int WorldToVoxelX(float x);
	int WorldToVoxelY(float y);
	int WorldToVoxelZ(float z);
	vec3 WorldToVoxel(const vec3 &World);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CRayTracer
{
private:
	BYTE *ColorBuffer;
	BITMAPINFO ColorBufferInfo;
	int Width, LineWidth, Height;

protected:
	CTriangle *Triangles;
	int TrianglesCount;

private:
	CUniformGrid UniformGrid;

public:
	bool SuperSampling;

public:
	CRayTracer();
	~CRayTracer();

	bool Init();
	void RayTrace(int x, int y);
	void Resize(int Width, int Height);
	void Destroy();

	void ClearColorBuffer();
	void SwapBuffers(HDC hDC);

protected:
	virtual bool InitScene() = 0;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CMyRayTracer : public CRayTracer
{
protected:
	bool InitScene();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CWnd
{
protected:
	char *WindowName;
	HWND hWnd;
	int Width, Height, x, y, LastX, LastY;

public:
	CWnd();
	~CWnd();

	bool Create(HINSTANCE hInstance, char *WindowName, int Width, int Height);
	void RePaint();
	void Show(bool Maximized = false);
	void MsgLoop();
	void Destroy();

	void OnKeyDown(UINT Key);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnPaint();
	void OnRButtonDown(int X, int Y);
	void OnSize(int Width, int Height);
};

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow);
