#include "opengl_21_tutorials_win32_framework.h"

// ----------------------------------------------------------------------------------------------------------------------------

CBuffer::CBuffer()
{
	SetDefaults();
}

CBuffer::~CBuffer()
{
	Empty();
}

void CBuffer::SetDefaults()
{
	Buffer = NULL;
	BufferSize = 0;

	Position = 0;
}

void CBuffer::AddData(void *Data, int DataSize)
{
	int Remaining = BufferSize - Position;

	if(DataSize > Remaining)
	{
		BYTE *OldBuffer = Buffer;
		int OldBufferSize = BufferSize;

		int Needed = DataSize - Remaining;

		BufferSize += Needed > BUFFER_SIZE_INCREMENT ? Needed : BUFFER_SIZE_INCREMENT;

		Buffer = new BYTE[BufferSize];

		memcpy(Buffer, OldBuffer, OldBufferSize);

		delete [] OldBuffer;
	}

	memcpy(Buffer + Position, Data, DataSize);

	Position += DataSize;
}

void CBuffer::Empty()
{
	if(Buffer != NULL)
	{
		delete [] Buffer;
	}

	SetDefaults();
}

void *CBuffer::GetData()
{
	return Buffer;
}

int CBuffer::GetDataSize()
{
	return Position;
}

// ----------------------------------------------------------------------------------------------------------------------------

int gl_max_texture_size = 0, gl_max_texture_max_anisotropy_ext = 0;

// ----------------------------------------------------------------------------------------------------------------------------

CTexture::CTexture()
{
	SetDefaults();
}

CTexture::~CTexture()
{
}

void CTexture::SetDefaults()
{
	Texture = 0;
}

CTexture::operator GLuint ()
{
	return Texture;
}

FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP)
{
	FREE_IMAGE_FORMAT fif = FreeImage_GetFileType(FileName);

	if(fif == FIF_UNKNOWN)
	{
		fif = FreeImage_GetFIFFromFilename(FileName);
	}

	if(fif == FIF_UNKNOWN)
	{
		return NULL;
	}

	FIBITMAP *dib = NULL;

	if(FreeImage_FIFSupportsReading(fif))
	{
		dib = FreeImage_Load(fif, FileName);
	}

	if(dib != NULL)
	{
		int OriginalWidth = FreeImage_GetWidth(dib);
		int OriginalHeight = FreeImage_GetHeight(dib);

		Width = OriginalWidth;
		Height = OriginalHeight;

		if(Width == 0 || Height == 0)
		{
			FreeImage_Unload(dib);
			return NULL;
		}

		BPP = FreeImage_GetBPP(dib);

		if(Width > gl_max_texture_size) Width = gl_max_texture_size;
		if(Height > gl_max_texture_size) Height = gl_max_texture_size;

		if(!GLEW_ARB_texture_non_power_of_two)
		{
			Width = 1 << (int)floor((log((float)Width) / log(2.0f)) + 0.5f); 
			Height = 1 << (int)floor((log((float)Height) / log(2.0f)) + 0.5f);
		}

		if(Width != OriginalWidth || Height != OriginalHeight)
		{
			FIBITMAP *rdib = FreeImage_Rescale(dib, Width, Height, FILTER_BICUBIC);
			FreeImage_Unload(dib);
			dib = rdib;
		}
	}

	return dib;
}

bool CTexture::LoadTexture2D(char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	int Width, Height, BPP;

	FIBITMAP *dib = GetBitmap(DirectoryFileName, Width, Height, BPP);

	if(dib == NULL)
	{
		ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
		return false;
	}

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileName);
		FreeImage_Unload(dib);
		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_2D, Texture);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib));

	glBindTexture(GL_TEXTURE_2D, 0);

	FreeImage_Unload(dib);

	return true;
}

bool CTexture::LoadTextureCubeMap(char **FileNames)
{
	int Width, Height, BPP;

	FIBITMAP *dib[6];

	bool Error = false;

	for(int i = 0; i < 6; i++)
	{
		CString DirectoryFileName = ModuleDirectory + FileNames[i];

		dib[i] = GetBitmap(DirectoryFileName, Width, Height, BPP);

		if(dib[i] == NULL)
		{
			ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
			Error = true;
		}
	}

	if(Error)
	{
		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileNames[5]);

		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_CUBE_MAP, Texture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_GENERATE_MIPMAP, GL_TRUE);

	for(int i = 0; i < 6; i++)
	{
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib[i]));
	}

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	for(int i = 0; i < 6; i++)
	{
		FreeImage_Unload(dib[i]);
	}

	return true;
}

void CTexture::Destroy()
{
	if(Texture != 0)
	{
		glDeleteTextures(1, &Texture);
	}

	SetDefaults();
}

// ----------------------------------------------------------------------------------------------------------------------------

CShaderProgram::CShaderProgram()
{
	SetDefaults();
}

CShaderProgram::~CShaderProgram()
{
}

void CShaderProgram::SetDefaults()
{
	VertexShader = 0;
	FragmentShader = 0;

	Program = 0;

	UniformLocations = NULL;
	AttribLocations = NULL;
}

CShaderProgram::operator GLuint ()
{
	return Program;
}

GLuint CShaderProgram::LoadShader(char *FileName, GLenum Type)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file " + DirectoryFileName + "!\r\n");
		return 0;
	}

	fseek(File, 0, SEEK_END);
	long Size = ftell(File);
	fseek(File, 0, SEEK_SET);
	char *Source = new char[Size + 1];
	fread(Source, 1, Size, File);
	fclose(File);
	Source[Size] = 0;

	GLuint Shader = glCreateShader(Type);

	glShaderSource(Shader, 1, (const char**)&Source, NULL);
	delete [] Source;
	glCompileShader(Shader);

	int CompileStatus;
	glGetShaderiv(Shader, GL_COMPILE_STATUS, &CompileStatus);

	if(CompileStatus == GL_FALSE)
	{
		ErrorLog.Append("Error compiling shader %s!\r\n", FileName);

		int InfoLogLength = 0;
		glGetShaderiv(Shader, GL_INFO_LOG_LENGTH, &InfoLogLength);

		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetShaderInfoLog(Shader, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		glDeleteShader(Shader);

		return 0;
	}

	return Shader;
}

bool CShaderProgram::Load(char *VertexShaderFileName, char *FragmentShaderFileName)
{
	bool Error = false;

	Destroy();

	Error |= ((VertexShader = LoadShader(VertexShaderFileName, GL_VERTEX_SHADER)) == 0);
	Error |= ((FragmentShader = LoadShader(FragmentShaderFileName, GL_FRAGMENT_SHADER)) == 0);

	if(Error)
	{
		Destroy();
		return false;
	}

	Program = glCreateProgram();
	glAttachShader(Program, VertexShader);
	glAttachShader(Program, FragmentShader);
	glLinkProgram(Program);

	int LinkStatus;
	glGetProgramiv(Program, GL_LINK_STATUS, &LinkStatus);

	if(LinkStatus == GL_FALSE)
	{
		ErrorLog.Append("Error linking program (%s, %s)!\r\n", VertexShaderFileName, FragmentShaderFileName);

		int InfoLogLength = 0;
		glGetProgramiv(Program, GL_INFO_LOG_LENGTH, &InfoLogLength);

		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetProgramInfoLog(Program, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		Destroy();

		return false;
	}

	return true;
}

void CShaderProgram::Destroy()
{
	if(Program != 0)
	{
		if(VertexShader != 0)
		{
			glDetachShader(Program, VertexShader);
		}

		if(FragmentShader != 0)
		{
			glDetachShader(Program, FragmentShader);
		}
	}

	if(VertexShader != 0)
	{
		glDeleteShader(VertexShader);
	}

	if(FragmentShader)
	{
		glDeleteShader(FragmentShader);
	}

	if(Program != 0)
	{
		glDeleteProgram(Program);
	}

	if(UniformLocations != NULL)
	{
		delete [] UniformLocations;
	}

	if(AttribLocations != NULL)
	{
		delete [] AttribLocations;
	}

	SetDefaults();
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera::CCamera()
{
	X = vec3(1.0f, 0.0f, 0.0f);
	Y = vec3(0.0f, 1.0f, 0.0f);
	Z = vec3(0.0f, 0.0f, 1.0f);

	Position = vec3(0.0f, 0.0f, 5.0f);
	Reference = vec3(0.0f, 0.0f, 0.0f);

	CalculateViewMatrix();
}

CCamera::~CCamera()
{
}

void CCamera::Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference)
{
	this->Position = Position;
	this->Reference = Reference;

	Z = normalize(Position - Reference);

	GetXY(Z, X, Y);

	if(!RotateAroundReference)
	{
		this->Reference = this->Position - Z * 0.05f;
	}

	CalculateViewMatrix();
}

void CCamera::Move(const vec3 &Movement)
{
	Position += Movement;
	Reference += Movement;

	CalculateViewMatrix();
}

vec3 CCamera::OnKeys(BYTE Keys, float FrameTime)
{
	float Speed = 5.0f;

	if(Keys & 0x40) Speed *= 2.0f;
	if(Keys & 0x80) Speed *= 0.5f;

	float Distance = Speed * FrameTime;

	vec3 Up(0.0f, 1.0f, 0.0f);
	vec3 Right = X;
	vec3 Forward = cross(Up, Right);

	Up *= Distance;
	Right *= Distance;
	Forward *= Distance;

	vec3 Movement;

	if(Keys & 0x01) Movement += Forward;
	if(Keys & 0x02) Movement -= Forward;
	if(Keys & 0x04) Movement -= Right;
	if(Keys & 0x08) Movement += Right;
	if(Keys & 0x10) Movement += Up;
	if(Keys & 0x20) Movement -= Up;

	return Movement;
}

void CCamera::OnMouseMove(int dx, int dy)
{
	float Sensitivity = 0.25f;

	Position -= Reference;

	if(dx != 0)
	{
		float DeltaX = (float)dx * Sensitivity;

		X = rotate(X, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Y = rotate(Y, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Z = rotate(Z, DeltaX, vec3(0.0f, 1.0f, 0.0f));
	}

	if(dy != 0)
	{
		float DeltaY = (float)dy * Sensitivity;

		Y = rotate(Y, DeltaY, X);
		Z = rotate(Z, DeltaY, X);

		if(Y.y < 0.0f)
		{
			Z = vec3(0.0f, Z.y > 0.0f ? 1.0f : -1.0f, 0.0f);
			Y = cross(Z, X);
		}
	}

	Position = Reference + Z * length(Position);

	CalculateViewMatrix();
}

void CCamera::OnMouseWheel(float zDelta)
{
	Position -= Reference;

	if(zDelta < 0 && length(Position) < 500.0f)
	{
		Position += Position * 0.1f;
	}

	if(zDelta > 0 && length(Position) > 0.05f)
	{
		Position -= Position * 0.1f;
	}

	Position += Reference;

	CalculateViewMatrix();
}

void CCamera::SetPerspective(float fovy, float aspect, float n, float f)
{
	ProjectionMatrix = perspective(fovy, aspect, n, f);
	ProjectionMatrixInverse = inverse(ProjectionMatrix);
	ViewProjectionMatrix = ProjectionMatrix * ViewMatrix;
	ViewProjectionMatrixInverse = ViewMatrixInverse * ProjectionMatrixInverse;
}

void CCamera::CalculateViewMatrix()
{
	ViewMatrix = mat4x4(X.x, Y.x, Z.x, 0.0f, X.y, Y.y, Z.y, 0.0f, X.z, Y.z, Z.z, 0.0f, -dot(X, Position), -dot(Y, Position), -dot(Z, Position), 1.0f);
	ViewMatrixInverse = inverse(ViewMatrix);
	ViewProjectionMatrix = ProjectionMatrix * ViewMatrix;
	ViewProjectionMatrixInverse = ViewMatrixInverse * ProjectionMatrixInverse;
}

// ----------------------------------------------------------------------------------------------------------------------------

CTriangle::CTriangle()
{
}

CTriangle::CTriangle(const vec3 &A, const vec3 &B, const vec3 &C)
{
	Set(A, B, C);
}

CTriangle::~CTriangle()
{
}

void CTriangle::Set(const vec3 &A, const vec3 &B, const vec3 &C)
{
	this->A = A;
	this->B = B;
	this->C = C;

	M = (A + B + C) / 3.0f;

	AB = B - A;
	BC = C - B;
	CA = A - C;

	LAB = length(AB);
	LBC = length(BC);
	LCA = length(CA);

	AB /= LAB;
	BC /= LBC;
	CA /= LCA;

	N = normalize(cross(AB, -CA));
	D = -dot(N, A);

	NH = (N.y > -1.0f && N.y < 1.0f) ? normalize(vec3(N.x, 0.0f, N.z)) : vec3(0.0f);
	NdotNH = dot(N, NH);

	N1 = normalize(cross(N, AB));
	D1 = -dot(N1, A);

	N2 = normalize(cross(N, BC));
	D2 = -dot(N2, B);

	N3 = normalize(cross(N, CA));
	D3 = -dot(N3, C);

	HPNAB = (AB.y > -1.0f && AB.y < 1.0f) ? normalize(vec3(0.0f, 1.0f, 0.0f) - AB * AB.y) : vec3(0.0f);
	HPDAB = -dot(A, HPNAB);
	VPNAB = cross(AB, HPNAB);
	VPDAB = -dot(A, VPNAB);

	HPNBC = (BC.y > -1.0f && BC.y < 1.0f) ? normalize(vec3(0.0f, 1.0f, 0.0f) - BC * BC.y) : vec3(0.0f);
	HPDBC = -dot(B, HPNBC);
	VPNBC = cross(BC, HPNBC);
	VPDBC = -dot(B, VPNBC);

	HPNCA = (CA.y > -1.0f && CA.y < 1.0f) ? normalize(vec3(0.0f, 1.0f, 0.0f) - CA * CA.y) : vec3(0.0f);
	HPDCA = -dot(C, HPNCA);
	VPNCA = cross(CA, HPNCA);
	VPDCA = -dot(C, VPNCA);
}

bool CTriangle::Inside(const vec3 &Point)
{
	if(dot(N1, Point) + D1 < 0.0f) return false;
	if(dot(N2, Point) + D2 < 0.0f) return false;
	if(dot(N3, Point) + D3 < 0.0f) return false;

	return true;
}

bool CTriangle::RayTriangleIntersectionTest(const vec3 &RayOrigin, const vec3 &RayDirection, float &MinDistance, vec3 &IntersectionPoint)
{
	float NdotRD = -dot(N, RayDirection);

	if(NdotRD > 0.0f)
	{
		float DistanceFromPlane = (dot(N, RayOrigin) + D) / NdotRD;

		if(DistanceFromPlane > 0.0f && DistanceFromPlane < MinDistance)
		{
			vec3 PointOnPlane = RayOrigin + RayDirection * DistanceFromPlane;

			if(Inside(PointOnPlane))
			{
				MinDistance = DistanceFromPlane;
				IntersectionPoint = PointOnPlane;

				return true;
			}
		}
	}

	return false;
}

bool CTriangle::GetHeightAbove(const vec3 &EyePosition, float &MinDistance, float &Height)
{
	float NdotRD = -N.y;

	if(NdotRD > 0.0f)
	{
		float DistanceFromPlane = (dot(N, EyePosition) + D) / NdotRD;

		if(DistanceFromPlane > 0.0f && DistanceFromPlane < MinDistance)
		{
			vec3 PointOnPlane = vec3(EyePosition.x, EyePosition.y + DistanceFromPlane, EyePosition.z);

			if(Inside(PointOnPlane))
			{
				MinDistance = DistanceFromPlane;
				Height = PointOnPlane.y;

				return true;
			}
		}
	}

	return false;
}

bool CTriangle::GetHeightUnder(const vec3 &EyePosition, float EyeKneeDistance, float &MinDistance, float &Height)
{
	float NdotRD = N.y;

	if(NdotRD > 0.0f)
	{
		float DistanceFromPlane = (dot(N, EyePosition) + D) / NdotRD;

		if(DistanceFromPlane > EyeKneeDistance && DistanceFromPlane < MinDistance)
		{
			vec3 PointOnPlane = vec3(EyePosition.x, EyePosition.y - DistanceFromPlane, EyePosition.z);

			if(Inside(PointOnPlane))
			{
				MinDistance = DistanceFromPlane;
				Height = PointOnPlane.y;

				return true;
			}
		}
	}

	return false;
}

bool CTriangle::IntersectionTest(const vec3 &EyePositionA, const vec3 &EyePositionB, const vec3 &Direction, float EyeKneeDistance, float ClosestDistance, const vec3 &PN, float PD, float &MinDistance, vec3 &Compensation)
{
	bool IntersectionTestPassed = false;

	if(NdotNH > 0.0f)
	{
		float NdotD = -dot(N, Direction);

		if(NdotD > 0.0f)
		{
			float DistanceFromPlane = (dot(N, EyePositionA) + D) / NdotD;

			if(DistanceFromPlane > 0.0f && DistanceFromPlane < MinDistance)
			{
				vec3 PointOnPlane = EyePositionA + Direction * DistanceFromPlane;

				if(Inside(PointOnPlane))
				{
					IntersectionTestPassed = true;
					MinDistance = DistanceFromPlane;
					Compensation = PointOnPlane - EyePositionB + NH * (ClosestDistance / NdotNH);
				}
			}
		}
	}

	vec3 *Vertices = (vec3*)&A;
	vec3 *Edges = (vec3*)&AB;
	float *EdgesLengths = &LAB;
	vec3 *VPNs = (vec3*)&VPNAB;

	for(int i = 0; i < 3; i++)
	{
		float PNdotE = -dot(PN, Edges[i]);

		if(PNdotE != 0.0f)
		{
			float DistanceFromPlane = (dot(PN, Vertices[i]) + PD) / PNdotE;

			if(DistanceFromPlane > 0.0f && DistanceFromPlane < EdgesLengths[i])
			{
				vec3 PointOnPlane = Vertices[i] + Edges[i] * DistanceFromPlane;

				vec3 EPAPOP = PointOnPlane - EyePositionA;

				float DistanceV = -EPAPOP.y;

				if(DistanceV > 0.0f && DistanceV < EyeKneeDistance)
				{
					float DistanceH = dot(Direction, EPAPOP);

					if(DistanceH > 0.0f && DistanceH < MinDistance)
					{
						IntersectionTestPassed = true;
						MinDistance = DistanceH;
						Compensation = vec3(PointOnPlane.x - EyePositionB.x, 0.0f, PointOnPlane.z - EyePositionB.z);
						float VPNdotD = -dot(VPNs[i], Direction);
						if(VPNdotD > 0.0f) Compensation += VPNs[i] * ClosestDistance;
						if(VPNdotD < 0.0f) Compensation -= VPNs[i] * ClosestDistance;
					}
				}
			}
		}
	}

	return IntersectionTestPassed;
}

bool CTriangle::DistanceTest(const vec3 &EyePositionB, float EyeKneeDistance, float ClosestDistance, float &MinDistance, vec3 &Compensation)
{
	bool DistanceTestFailed = false;

	if(NdotNH > 0.0f)
	{
		float DistanceFromPlane = dot(N, EyePositionB) + D;

		if(DistanceFromPlane > 0.0f && DistanceFromPlane < MinDistance)
		{
			if(Inside(EyePositionB))
			{
				DistanceTestFailed = true;
				MinDistance = DistanceFromPlane;
				Compensation = NH * ((ClosestDistance - DistanceFromPlane) / NdotNH);
			}
		}
	}

	vec3 *Vertices = (vec3*)&A;
	vec3 *Edges = (vec3*)&AB;
	float *EdgesLengths = &LAB;

	for(int i = 0; i < 3; i++)
	{
		vec3 EPBD = EyePositionB - Vertices[i];

		float EdotEPBD = dot(Edges[i], EPBD);

		if(EdotEPBD > 0.0f && EdotEPBD < EdgesLengths[i])
		{
			vec3 N = EPBD - Edges[i] * EdotEPBD;

			if(N.x != 0.0f || N.z != 0.0f)
			{
				float DistanceFromEdge = length(N);

				if(DistanceFromEdge > 0.0f && DistanceFromEdge < MinDistance)
				{
					DistanceTestFailed = true;
					MinDistance = DistanceFromEdge;
					N /= DistanceFromEdge;
					vec3 NH = normalize(vec3(N.x, 0.0f, N.z));
					float NdotNH = dot(N, NH);
					Compensation = NH * ((ClosestDistance - DistanceFromEdge) / NdotNH);
				}
			}
		}
	}

	for(int i = 0; i < 3; i++)
	{
		vec3 N = EyePositionB - Vertices[i];

		if(N.x != 0.0f || N.z != 0.0f)
		{
			float DistanceFromVertex = length(N);

			if(DistanceFromVertex > 0.0f && DistanceFromVertex < MinDistance)
			{
				DistanceTestFailed = true;
				MinDistance = DistanceFromVertex;
				N /= DistanceFromVertex;
				vec3 NH = normalize(vec3(N.x, 0.0f, N.z));
				float NdotNH = dot(N, NH);
				Compensation = NH * ((ClosestDistance - DistanceFromVertex) / NdotNH);
			}
		}
	}

	vec3 *HPNs = (vec3*)&HPNAB;
	float *HPDs = &HPDAB;
	vec3 *VPNs = (vec3*)&VPNAB;
	float *VPDs = &VPDAB;

	for(int i = 0; i < 3; i++)
	{
		if(HPNs[i].y > 0.0f)
		{
			float DistanceFromHorizontalPlane = (dot(HPNs[i], EyePositionB) + HPDs[i]) / HPNs[i].y;

			if(DistanceFromHorizontalPlane > 0.0f && DistanceFromHorizontalPlane < EyeKneeDistance)
			{
				float DistanceFromVerticalPlane = dot(VPNs[i], EyePositionB) + VPDs[i];

				if(DistanceFromVerticalPlane > 0.0f && DistanceFromVerticalPlane < MinDistance)
				{
					vec3 PointOnHorizontalPlane = vec3(EyePositionB.x, EyePositionB.y - DistanceFromHorizontalPlane, EyePositionB.z);

					float EdotPOHPD = dot(Edges[i], PointOnHorizontalPlane - Vertices[i]);

					if(EdotPOHPD > 0.0f && EdotPOHPD < EdgesLengths[i])
					{
						DistanceTestFailed = true;
						MinDistance = DistanceFromVerticalPlane;
						Compensation = VPNs[i] * (ClosestDistance - DistanceFromVerticalPlane);
					}
				}
			}
		}
	}

	for(int i = 0; i < 3; i++)
	{
		vec3 EPBD = Vertices[i] - EyePositionB;

		float EdotEPBD = -EPBD.y;

		if(EdotEPBD > 0.0f && EdotEPBD < EyeKneeDistance)
		{
			vec3 N = vec3(EPBD.x, EPBD.y + EdotEPBD, EPBD.z);

			float DistanceFromVertex = length(N);

			if(DistanceFromVertex > 0.0f && DistanceFromVertex < MinDistance)
			{
				DistanceTestFailed = true;
				MinDistance = DistanceFromVertex;
				N /= DistanceFromVertex;
				Compensation = N * (DistanceFromVertex - ClosestDistance);
			}
		}
	}

	return DistanceTestFailed;
}

// ----------------------------------------------------------------------------------------------------------------------------

CCollisionDetector::CCollisionDetector()
{
	SetDefaults();
}

CCollisionDetector::~CCollisionDetector()
{
}

void CCollisionDetector::SetDefaults()
{
	Triangles = NULL;
	TrianglesCount = 0;

	EyeHeight = 0.0f;
	EyeKneeDistance = 0.0f;
	ClosestDistance = 0.0f;

	EH = 0.0f;
	EHD2 = 0.0f;
	EKD = 0.0f;
	EKDD2 = 0.0f;

	FallSpeed = 0.0f;
	CrouchState = 0;
}

void CCollisionDetector::Init(vec3 *Vertices, int VerticesCount, float EyeHeight, float EyeKneeDistance, float ClosestDistance)
{
	Destroy();

	this->EyeHeight = EyeHeight;
	this->EyeKneeDistance = EyeKneeDistance;
	this->ClosestDistance = ClosestDistance;

	EH = EyeHeight;
	EHD2 = EyeHeight / 2.0f;
	EKD = EyeKneeDistance;
	EKDD2 = EyeKneeDistance / 2.0f;

	if(Vertices != NULL && VerticesCount > 0)
	{
		TrianglesCount = VerticesCount / 3;

		Triangles = new CTriangle[TrianglesCount];

		for(int i = 0; i < TrianglesCount; i++)
		{
			Triangles[i].Set(Vertices[i * 3 + 0], Vertices[i * 3 + 1], Vertices[i * 3 + 2]);
		}
	}
}

void CCollisionDetector::Destroy()
{
	if(Triangles != NULL)
	{
		delete [] Triangles;
	}

	SetDefaults();
}

void CCollisionDetector::Jump()
{
	if(CrouchState == 0)
	{
		if(FallSpeed == 0.0f)
		{
			FallSpeed = -9.82f / 3.0f;
		}
	}
	else
	{
		CrouchState = 2;
	}
}

void CCollisionDetector::Crouch()
{
	if(CrouchState == 0)
	{
		EyeHeight = EHD2;
		EyeKneeDistance = EKDD2;
		CrouchState = 1;
	}
	else if(FallSpeed < 0.0f)
	{
		if(CrouchState == 1)
		{
			EyeHeight = EH;
			EyeKneeDistance = EKD;
			CrouchState = 0;
		}
	}
	else
	{
		if(CrouchState == 1)
		{
			CrouchState = 2;
		}
		else if(CrouchState == 2)
		{
			EyeHeight = EHD2;
			EyeKneeDistance = EKDD2;
			CrouchState = 1;
		}
	}
}

bool CCollisionDetector::GetHeightAbove(const vec3 &EyePositionA, float &MinDistance, float &Height)
{
	bool HeightFound = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		HeightFound |= Triangles[i].GetHeightAbove(EyePositionA, MinDistance, Height);
	}

	return HeightFound;
}

bool CCollisionDetector::GetHeightUnder(const vec3 &EyePositionA, float EyeKneeDistance, float &MinDistance, float &Height)
{
	bool HeightFound = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		HeightFound |= Triangles[i].GetHeightUnder(EyePositionA, EyeKneeDistance, MinDistance, Height);
	}

	return HeightFound;
}

bool CCollisionDetector::IntersectionTest(const vec3 &EyePositionA, const vec3 &EyePositionB, const vec3 &Direction, float EyeKneeDistance, float ClosestDistance, const vec3 &PN, float PD, float &MinDistance, vec3 &Compensation)
{
	bool IntersectionTestPassed = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		IntersectionTestPassed |= Triangles[i].IntersectionTest(EyePositionA, EyePositionB, Direction, EyeKneeDistance, ClosestDistance, PN, PD, MinDistance, Compensation);
	}

	return IntersectionTestPassed;
}

bool CCollisionDetector::DistanceTest(const vec3 &EyePositionB, float EyeKneeDistance, float ClosestDistance, float &MinDistance, vec3 &Compensation)
{
	bool DistanceTestFailed = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		DistanceTestFailed |= Triangles[i].DistanceTest(EyePositionB, EyeKneeDistance, ClosestDistance, MinDistance, Compensation);
	}

	return DistanceTestFailed;
}

void CCollisionDetector::CheckHorizontalCollision(const vec3 &EyePosition, vec3 &Movement)
{
	if(CrouchState != 0)
	{
		Movement *= 0.5f;
	}

	int Depth = 0;

	TestAgain:

	if(Depth < 16)
	{
		vec3 EyePositionA = EyePosition;
		float Length = length(Movement);
		vec3 Direction = Movement / Length;
		vec3 EyePositionB = EyePositionA + Movement;

		if(Length > ClosestDistance)
		{
			vec3 PN = cross(Direction, vec3(0.0f, -1.0f, 0.0f));
			float PD = -dot(PN, EyePositionA);
			float Distance = Length;
			vec3 Compensation;

			if(IntersectionTest(EyePositionA, EyePositionB, Direction, EyeKneeDistance, ClosestDistance, PN, PD, Distance, Compensation))
			{
				Movement += Compensation;

				Depth++;

				goto TestAgain;
			}
		}

		float Distance = ClosestDistance;
		vec3 Compensation;

		if(DistanceTest(EyePositionB, EyeKneeDistance, ClosestDistance, Distance, Compensation))
		{
			Movement += Compensation;

			Depth++;

			goto TestAgain;
		}
	}
}

void CCollisionDetector::CheckVerticalCollision(const vec3 &EyePosition, float FrameTime, vec3 &Movement)
{
	if(CrouchState == 2)
	{
		float DistanceAbove = EH - EyeHeight + ClosestDistance, HeightAbove;

		if(!GetHeightAbove(EyePosition, DistanceAbove, HeightAbove))
		{
			EyeHeight += EH * 2.0f * FrameTime;
			EyeKneeDistance = EyeHeight * EKD / EH;

			if(EyeHeight >= EH)
			{
				EyeHeight = EH;
				EyeKneeDistance = EKD;
				CrouchState = 0;
			}
		}
	}

	float DistanceUnder = 1048576.0f, HeightUnder = 0.0f;

	GetHeightUnder(EyePosition, EyeKneeDistance, DistanceUnder, HeightUnder);

	float EPYMEH = EyePosition.y - EyeHeight;

	if(HeightUnder < EPYMEH || FallSpeed < 0.0f)
	{
		FallSpeed += 9.82f * FrameTime;

		float Distance = FallSpeed * FrameTime;

		if(FallSpeed < 0.0f)
		{
			float DistanceAbove = ClosestDistance - Distance, HeightAbove;

			if(GetHeightAbove(EyePosition, DistanceAbove, HeightAbove))
			{
				Distance = DistanceAbove - ClosestDistance;
				FallSpeed = 0.0f;
			}
		}

		float EPYMEHMHU = EPYMEH - HeightUnder;

		if(Distance > EPYMEHMHU)
		{
			Distance = EPYMEHMHU;
		}

		Movement = vec3(0.0f, -Distance, 0.0f);
	}
	else
	{
		FallSpeed = 0.0f;

		float HUMEPYMEH = HeightUnder - EPYMEH;

		if(HUMEPYMEH < EyeHeight - EyeKneeDistance)
		{
			Movement = vec3(0.0f, HUMEPYMEH, 0.0f);
		}
	}

	if(Movement.y != 0.0f)
	{
		int Depth = 0;

		TestAgain:

		if(Depth < 16)
		{
			float Distance = ClosestDistance;
			vec3 Compensation;

			if(DistanceTest(EyePosition + Movement, EyeKneeDistance, ClosestDistance, Distance, Compensation))
			{
				Movement += Compensation;

				Depth++;

				goto TestAgain;
			}
		}
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

CScene::CScene()
{
	SetDefaults();
}

CScene::~CScene()
{
}

void CScene::SetDefaults()
{
	Vertices = NULL;
	VerticesCount = 0;

	VertexBufferObject = 0;
}

bool CScene::LoadBinary(const char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error opening file " + DirectoryFileName + "!\r\n");
		return false;
	}

	Destroy();

	if(fread(&VerticesCount, sizeof(int), 1, File) != 1)
	{
		ErrorLog.Append("Error reading file " + DirectoryFileName + "!\r\n");
		fclose(File);
		return false;
	}

	if(VerticesCount > 0)
	{
		Vertices = new vec3[VerticesCount];

		if(fread(Vertices, sizeof(vec3), VerticesCount, File) != VerticesCount)
		{
			ErrorLog.Append("Error reading file " + DirectoryFileName + "!\r\n");
			fclose(File);
			Destroy();
			return false;
		}

		vec3 *VertexBufferData = new vec3[VerticesCount * 3];

		for(int i = 0; i < VerticesCount; i += 3)
		{
			vec3 VertexA = Vertices[i + 0];
			vec3 VertexB = Vertices[i + 1];
			vec3 VertexC = Vertices[i + 2];

			vec3 Normal = normalize(cross(VertexB - VertexA, VertexC - VertexA));

			mat3x3 TBN = GetTBNMatrix(Normal);

			vec3 TexCoordA = TBN * VertexA;
			vec3 TexCoordB = TBN * VertexB;
			vec3 TexCoordC = TBN * VertexC;

			VertexBufferData[i * 3 + 0] = VertexA;
			VertexBufferData[i * 3 + 1] = Normal;
			VertexBufferData[i * 3 + 2] = TexCoordA;
			VertexBufferData[i * 3 + 3] = VertexB;
			VertexBufferData[i * 3 + 4] = Normal;
			VertexBufferData[i * 3 + 5] = TexCoordB;
			VertexBufferData[i * 3 + 6] = VertexC;
			VertexBufferData[i * 3 + 7] = Normal;
			VertexBufferData[i * 3 + 8] = TexCoordC;
		}

		glGenBuffers(1, &VertexBufferObject);

		glBindBuffer(GL_ARRAY_BUFFER, VertexBufferObject);
		glBufferData(GL_ARRAY_BUFFER, VerticesCount * 3 * 12, VertexBufferData, GL_STATIC_DRAW);
		glBindBuffer(GL_ARRAY_BUFFER, 0);

		delete [] VertexBufferData;
	}

	fclose(File);

	if(!Texture.LoadTexture2D("concrete.jpg"))
	{
		Destroy();
		return false;
	}

	return true;
}

void CScene::Render()
{
	glBindBuffer(GL_ARRAY_BUFFER, VertexBufferObject);

	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 36, (void*)0);

	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(GL_FLOAT, 36, (void*)12);

	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(3, GL_FLOAT, 36, (void*)24);

	glBindTexture(GL_TEXTURE_2D, Texture);

	glDrawArrays(GL_TRIANGLES, 0, VerticesCount);

	glBindTexture(GL_TEXTURE_2D, 0);

	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);

	glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void CScene::Destroy()
{
	if(Vertices != NULL)
	{
		delete [] Vertices;
	}

	if(VertexBufferObject != 0)
	{
		glDeleteBuffers(1, &VertexBufferObject);
	}

	Texture.Destroy();

	SetDefaults();
}

vec3 *CScene::GetVertices()
{
	return Vertices;
}

int CScene::GetVerticesCount()
{
	return VerticesCount;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer::COpenGLRenderer()
{
	Texturing = true;
	Lighting = true;
	ApplySSAO = true;
	ApplyFXAA = true;
}

COpenGLRenderer::~COpenGLRenderer()
{
}

bool COpenGLRenderer::Init()
{
	bool Error = false;

	if(!GLEW_ARB_texture_non_power_of_two)
	{
		ErrorLog.Append("GL_ARB_texture_non_power_of_two not supported!\r\n");
		Error = true;
	}

	if(!GLEW_ARB_depth_texture)
	{
		ErrorLog.Append("GLEW_ARB_depth_texture not supported!\r\n");
		Error = true;
	}

	if(!GLEW_EXT_framebuffer_object)
	{
		ErrorLog.Append("GL_EXT_framebuffer_object not supported!\r\n");
		Error = true;
	}

	Error |= !Preprocess.Load("preprocess.vs", "preprocess.fs");
	Error |= !SSAO.Load("ssao.vs", "ssao.fs");
	Error |= !SSAOFilterH.Load("ssaofilter.vs", "ssaofilterh.fs");
	Error |= !SSAOFilterV.Load("ssaofilter.vs", "ssaofilterv.fs");
	Error |= !DeferredLighting.Load("deferredlighting.vs", "deferredlighting.fs");
	Error |= !FXAA.Load("FXAA.vert", "FXAA_Extreme_Quality.frag");

	Error |= !Scene.LoadBinary("scene.bin");

	if(Error)
	{
		return false;
	}

	Preprocess.UniformLocations = new GLuint[1];
	Preprocess.UniformLocations[0] = glGetUniformLocation(Preprocess, "Texturing");

	SSAO.UniformLocations = new GLuint[2];
	SSAO.UniformLocations[0] = glGetUniformLocation(SSAO, "Scale");
	SSAO.UniformLocations[1] = glGetUniformLocation(SSAO, "ProjectionBiasMatrixInverse");

	SSAOFilterH.UniformLocations = new GLuint[1];
	SSAOFilterH.UniformLocations[0] = glGetUniformLocation(SSAOFilterH, "PixelSizeX");

	SSAOFilterV.UniformLocations = new GLuint[1];
	SSAOFilterV.UniformLocations[0] = glGetUniformLocation(SSAOFilterV, "PixelSizeY");

	DeferredLighting.UniformLocations = new GLuint[3];
	DeferredLighting.UniformLocations[0] = glGetUniformLocation(DeferredLighting, "ProjectionBiasMatrixInverse");
	DeferredLighting.UniformLocations[1] = glGetUniformLocation(DeferredLighting, "Lighting");
	DeferredLighting.UniformLocations[2] = glGetUniformLocation(DeferredLighting, "ApplySSAO");

	FXAA.UniformLocations = new GLuint[1];
	FXAA.UniformLocations[0] = glGetUniformLocation(FXAA, "RCPFrame");

	glUseProgram(SSAO);
	glUniform1i(glGetUniformLocation(SSAO, "NormalBuffer"), 0);
	glUniform1i(glGetUniformLocation(SSAO, "DepthBuffer"), 1);
	glUniform1i(glGetUniformLocation(SSAO, "RotationTexture"), 2);
	glUniform1f(glGetUniformLocation(SSAO, "Radius"), 0.125f);
	glUniform1f(glGetUniformLocation(SSAO, "Strength"), 2.0f);
	glUniform1f(glGetUniformLocation(SSAO, "ConstantAttenuation"), 1.0f);
	glUniform1f(glGetUniformLocation(SSAO, "LinearAttenuation"), 1.0f);
	glUniform1f(glGetUniformLocation(SSAO, "QuadraticAttenuation"), 0.0f);
	glUseProgram(0);

	float s = 128.0f, e = 131070.0f, fs = 1.0f / s, fe = 1.0f / e, fd = fs - fe;

	glUseProgram(SSAOFilterH);
	glUniform1i(glGetUniformLocation(SSAOFilterH, "SSAOBuffer"), 0);
	glUniform1i(glGetUniformLocation(SSAOFilterH, "DepthBuffer"), 1);
	glUniform1f(glGetUniformLocation(SSAOFilterH, "fs"), fs);
	glUniform1f(glGetUniformLocation(SSAOFilterH, "fd"), fd);
	glUseProgram(0);

	glUseProgram(SSAOFilterV);
	glUniform1i(glGetUniformLocation(SSAOFilterV, "SSAOBuffer"), 0);
	glUniform1i(glGetUniformLocation(SSAOFilterV, "DepthBuffer"), 1);
	glUniform1f(glGetUniformLocation(SSAOFilterV, "fs"), fs);
	glUniform1f(glGetUniformLocation(SSAOFilterV, "fd"), fd);
	glUseProgram(0);

	glUseProgram(DeferredLighting);
	glUniform1i(glGetUniformLocation(DeferredLighting, "ColorBuffer"), 0);
	glUniform1i(glGetUniformLocation(DeferredLighting, "NormalBuffer"), 1);
	glUniform1i(glGetUniformLocation(DeferredLighting, "DepthBuffer"), 2);
	glUniform1i(glGetUniformLocation(DeferredLighting, "SSAOBuffer"), 3);
	glUseProgram(0);

	srand(GetTickCount());

	vec2 *Samples = new vec2[16];
	float Angle = (float)M_PI_4;

	for(int i = 0; i < 16; i++)
	{
		Samples[i].x = cos(Angle) * (float)(i + 1) / 16.0f;
		Samples[i].y = sin(Angle) * (float)(i + 1) / 16.0f;

		Angle += (float)M_PI_2;

		if(((i + 1) % 4) == 0) Angle += (float)M_PI_4;
	}

	glUseProgram(SSAO);
	glUniform2fv(glGetUniformLocation(SSAO, "Samples"), 16, (float*)Samples);
	glUseProgram(0);

	delete [] Samples;

	vec4 *RotationTextureData = new vec4[64 * 64];
	float RandomAngle = (float)rand() / (float)RAND_MAX * (float)M_PI * 2.0f;

	for(int i = 0; i < 64 * 64; i++)
	{
		RotationTextureData[i].x = cos(RandomAngle) * 0.5f + 0.5f;
		RotationTextureData[i].y = sin(RandomAngle) * 0.5f + 0.5f;
		RotationTextureData[i].z = -sin(RandomAngle) * 0.5f + 0.5f;
		RotationTextureData[i].w = cos(RandomAngle) * 0.5f + 0.5f;

		RandomAngle += (float)rand() / (float)RAND_MAX * (float)M_PI * 2.0f;
	}

	glGenTextures(1, &RotationTexture);
	glBindTexture(GL_TEXTURE_2D, RotationTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, 64, 64, 0, GL_RGBA, GL_FLOAT, (float*)RotationTextureData);
	glBindTexture(GL_TEXTURE_2D, 0);

	delete [] RotationTextureData;

	glGenTextures(1, &ColorBuffer);
	glGenTextures(1, &NormalBuffer);
	glGenTextures(1, &DepthBuffer);
	glGenTextures(1, &SSAOBuffer);
	glGenTextures(1, &SSAOFilterBuffer);
	glGenTextures(1, &FXAABuffer);

	glGenFramebuffersEXT(1, &FBO);

	Camera.Look(vec3(0.0f, 1.75f, 7.0f), vec3(0.0f, 1.75f, 0.0f));

	CollisionDetector.Init(Scene.GetVertices(), Scene.GetVerticesCount(), 1.75f, 1.25f, 0.125f);

	return true;
}

void COpenGLRenderer::Render()
{
	GLenum Buffers[] = {GL_COLOR_ATTACHMENT0_EXT, GL_COLOR_ATTACHMENT1_EXT};

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
	glDrawBuffers(2, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, ColorBuffer, 0);
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, NormalBuffer, 0);
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, DepthBuffer, 0);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);

	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&Camera.ViewMatrix);

	glUseProgram(Preprocess);
	glUniform1i(Preprocess.UniformLocations[0], Texturing);

	Scene.Render();

	glUseProgram(0);

	glDisable(GL_CULL_FACE);
	glDisable(GL_DEPTH_TEST);

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	if(ApplySSAO)
	{
		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glDrawBuffers(1, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, SSAOBuffer, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, NormalBuffer);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, DepthBuffer);
		glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, RotationTexture);
		glUseProgram(SSAO);
		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();
		glUseProgram(0);
		glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glDrawBuffers(1, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, SSAOFilterBuffer, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, SSAOBuffer);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, DepthBuffer);
		glUseProgram(SSAOFilterH);
		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();
		glUseProgram(0);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glDrawBuffers(1, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, SSAOBuffer, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, SSAOFilterBuffer);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, DepthBuffer);
		glUseProgram(SSAOFilterV);
		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();
		glUseProgram(0);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
	}

	if(ApplyFXAA)
	{
		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glDrawBuffers(1, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, FXAABuffer, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);
	}

	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, ColorBuffer);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, NormalBuffer);
	glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, DepthBuffer);
	glActiveTexture(GL_TEXTURE3); glBindTexture(GL_TEXTURE_2D, SSAOBuffer);
	glUseProgram(DeferredLighting);
	glUniform1i(DeferredLighting.UniformLocations[1], Lighting);
	glUniform1i(DeferredLighting.UniformLocations[2], ApplySSAO);
	glBegin(GL_QUADS);
		glVertex2f(0.0f, 0.0f);
		glVertex2f(1.0f, 0.0f);
		glVertex2f(1.0f, 1.0f);
		glVertex2f(0.0f, 1.0f);
	glEnd();
	glUseProgram(0);
	glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

	if(ApplyFXAA)
	{
		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
	}

	if(ApplyFXAA)
	{
		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, FXAABuffer);
		glUseProgram(FXAA);
		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();
		glUseProgram(0);
		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);
	}
}

void COpenGLRenderer::Animate(float FrameTime)
{
}

void COpenGLRenderer::Resize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	glViewport(0, 0, Width, Height);

	Camera.SetPerspective(45.0f, (float)Width / (float)Height, 0.125f, 512.0f);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&Camera.ProjectionMatrix);

	mat4x4 ProjectionBiasMatrixInverse = Camera.ProjectionMatrixInverse * BiasMatrixInverse;

	glBindTexture(GL_TEXTURE_2D, ColorBuffer);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	glBindTexture(GL_TEXTURE_2D, NormalBuffer);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	glBindTexture(GL_TEXTURE_2D, DepthBuffer);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT24, Width, Height, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	glBindTexture(GL_TEXTURE_2D, SSAOBuffer);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	glBindTexture(GL_TEXTURE_2D, SSAOFilterBuffer);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	glBindTexture(GL_TEXTURE_2D, FXAABuffer);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	glUseProgram(SSAO);
	glUniform2f(SSAO.UniformLocations[0], (float)Width / 64.0f, (float)Height / 64.0f);
	glUniformMatrix4fv(SSAO.UniformLocations[1], 1, GL_FALSE, &ProjectionBiasMatrixInverse);
	glUseProgram(0);

	glUseProgram(SSAOFilterH);
	glUniform1f(SSAOFilterH.UniformLocations[0], 1.0f / (float)Width);
	glUseProgram(SSAOFilterV);
	glUniform1f(SSAOFilterV.UniformLocations[0], 1.0f / (float)Height);
	glUseProgram(0);

	glUseProgram(DeferredLighting);
	glUniformMatrix4fv(DeferredLighting.UniformLocations[0], 1, GL_FALSE, &ProjectionBiasMatrixInverse);
	glUseProgram(0);

	glUseProgram(FXAA);
	glUniform2f(FXAA.UniformLocations[0], 1.0f / (float)Width, 1.0f / (float)Height);
	glUseProgram(0);
}

void COpenGLRenderer::Destroy()
{
	Preprocess.Destroy();
	SSAO.Destroy();
	SSAOFilterH.Destroy();
	SSAOFilterV.Destroy();
	DeferredLighting.Destroy();
	FXAA.Destroy();

	Scene.Destroy();

	glDeleteTextures(1, &RotationTexture);

	glDeleteTextures(1, &ColorBuffer);
	glDeleteTextures(1, &NormalBuffer);
	glDeleteTextures(1, &DepthBuffer);
	glDeleteTextures(1, &SSAOBuffer);
	glDeleteTextures(1, &SSAOFilterBuffer);
	glDeleteTextures(1, &FXAABuffer);

	if(GLEW_EXT_framebuffer_object)
	{
		glDeleteFramebuffersEXT(1, &FBO);
	}

	CollisionDetector.Destroy();
}

void COpenGLRenderer::CheckCameraKeys(float FrameTime)
{
	BYTE Keys = 0x00;

	if(GetKeyState('W') & 0x80) Keys |= 0x01;
	if(GetKeyState('S') & 0x80) Keys |= 0x02;
	if(GetKeyState('A') & 0x80) Keys |= 0x04;
	if(GetKeyState('D') & 0x80) Keys |= 0x08;
	// if(GetKeyState('R') & 0x80) Keys |= 0x10;
	// if(GetKeyState('F') & 0x80) Keys |= 0x20;

	if(GetKeyState(VK_SHIFT) & 0x80) Keys |= 0x40;
	if(GetKeyState(VK_CONTROL) & 0x80) Keys |= 0x80;

	if(Keys & 0x3F)
	{
		vec3 Movement = Camera.OnKeys(Keys, FrameTime * 0.5f);

		CollisionDetector.CheckHorizontalCollision(Camera.Reference, Movement);

		if(length(Movement) > 0.0f)
		{
			Camera.Move(Movement);
		}
	}

	vec3 Movement;

	CollisionDetector.CheckVerticalCollision(Camera.Reference, FrameTime, Movement);

	if(length(Movement) > 0.0f)
	{
		Camera.Move(Movement);
	}
}

void COpenGLRenderer::OnKeyDown(UINT Key)
{
	switch(Key)
	{
		case VK_F1:
			Texturing = !Texturing;
			break;

		case VK_F2:
			Lighting = !Lighting;
			break;

		case VK_F3:
			ApplySSAO = !ApplySSAO;
			break;

		case VK_F4:
			ApplyFXAA = !ApplyFXAA;
			break;

		case 'C':
			CollisionDetector.Crouch();
			break;

		case VK_SPACE:
			CollisionDetector.Jump();
			break;
	}
}

void COpenGLRenderer::OnLButtonDown(int X, int Y)
{
	LastClickedX = X;
	LastClickedY = Y;
}

void COpenGLRenderer::OnLButtonUp(int X, int Y)
{
	if(X == LastClickedX && Y == LastClickedY)
	{
	}
}

void COpenGLRenderer::OnMouseMove(int X, int Y)
{
	if(GetKeyState(VK_RBUTTON) & 0x80)
	{
		Camera.OnMouseMove(LastX - X, LastY - Y);
	}

	LastX = X;
	LastY = Y;
}

void COpenGLRenderer::OnMouseWheel(short zDelta)
{
	// Camera.OnMouseWheel(zDelta);
}

void COpenGLRenderer::OnRButtonDown(int X, int Y)
{
	LastClickedX = X;
	LastClickedY = Y;
}

void COpenGLRenderer::OnRButtonUp(int X, int Y)
{
	if(X == LastClickedX && Y == LastClickedY)
	{
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

void GetModuleDirectory()
{
	char *moduledirectory = new char[256];
	GetModuleFileName(GetModuleHandle(NULL), moduledirectory, 256);
	*(strrchr(moduledirectory, '\\') + 1) = 0;
	ModuleDirectory = moduledirectory;
	delete [] moduledirectory;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView::COpenGLView()
{
}

COpenGLView::~COpenGLView()
{
}

bool COpenGLView::Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples)
{
	this->Title = Title;
	this->Width = Width;
	this->Height = Height;

	WNDCLASSEX WndClassEx;

	memset(&WndClassEx, 0, sizeof(WNDCLASSEX));

	WndClassEx.cbSize = sizeof(WNDCLASSEX);
	WndClassEx.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	WndClassEx.lpfnWndProc = WndProc;
	WndClassEx.hInstance = hInstance;
	WndClassEx.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hCursor = LoadCursor(NULL, IDC_ARROW);
	WndClassEx.lpszClassName = "Win32OpenGLWindowClass";

	if(RegisterClassEx(&WndClassEx) == 0)
	{
		ErrorLog.Set("RegisterClassEx failed!");
		return false;
	}

	DWORD Style = WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	hWnd = CreateWindowEx(WS_EX_APPWINDOW, WndClassEx.lpszClassName, Title, Style, 0, 0, Width, Height, NULL, NULL, hInstance, NULL);

	if(hWnd == NULL)
	{
		ErrorLog.Set("CreateWindowEx failed!");
		return false;
	}

	HDC hDC = GetDC(hWnd);

	if(hDC == NULL)
	{
		ErrorLog.Set("GetDC failed!");
		return false;
	}

	PIXELFORMATDESCRIPTOR pfd;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 32;
	pfd.cDepthBits = 24;
	pfd.iLayerType = PFD_MAIN_PLANE;

	int PixelFormat = ChoosePixelFormat(hDC, &pfd);

	if(PixelFormat == 0)
	{
		ErrorLog.Set("ChoosePixelFormat failed!");
		return false;
	}

	static int MSAAPixelFormat = 0;

	if(SetPixelFormat(hDC, MSAAPixelFormat == 0 ? PixelFormat : MSAAPixelFormat, &pfd) == FALSE)
	{
		ErrorLog.Set("SetPixelFormat failed!");
		return false;
	}

	hGLRC = wglCreateContext(hDC);

	if(hGLRC == NULL)
	{
		ErrorLog.Set("wglCreateContext failed!");
		return false;
	}

	if(wglMakeCurrent(hDC, hGLRC) == FALSE)
	{
		ErrorLog.Set("wglMakeCurrent failed!");
		return false;
	}

	if(glewInit() != GLEW_OK)
	{
		ErrorLog.Set("glewInit failed!");
		return false;
	}

	if(!GLEW_VERSION_2_1)
	{
		ErrorLog.Set("OpenGL 2.1 not supported!");
		return false;
	}

	if(MSAAPixelFormat == 0 && Samples > 0)
	{
		if(GLEW_ARB_multisample && WGLEW_ARB_pixel_format)
		{
			while(Samples > 0)
			{
				UINT NumFormats = 0;

				int PFAttribs[] =
				{
					WGL_DRAW_TO_WINDOW_ARB, GL_TRUE,
					WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
					WGL_DOUBLE_BUFFER_ARB, GL_TRUE,
					WGL_PIXEL_TYPE_ARB, WGL_TYPE_RGBA_ARB,
					WGL_COLOR_BITS_ARB, 32,
					WGL_DEPTH_BITS_ARB, 24,
					WGL_ACCELERATION_ARB, WGL_FULL_ACCELERATION_ARB,
					WGL_SAMPLE_BUFFERS_ARB, GL_TRUE,
					WGL_SAMPLES_ARB, Samples,
					0
				};

				if(wglChoosePixelFormatARB(hDC, PFAttribs, NULL, 1, &MSAAPixelFormat, &NumFormats) == TRUE && NumFormats > 0) break;

				Samples--;
			}

			wglDeleteContext(hGLRC);
			DestroyWindow(hWnd);
			UnregisterClass(WndClassEx.lpszClassName, hInstance);

			return Init(hInstance, Title, Width, Height, Samples);
		}
		else
		{
			Samples = 0;
		}
	}

	this->Samples = Samples;

	GetModuleDirectory();

	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &gl_max_texture_size);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &gl_max_texture_max_anisotropy_ext);
	}

	if(WGLEW_EXT_swap_control)
	{
		wglSwapIntervalEXT(0);
	}

	return OpenGLRenderer.Init();
}

void COpenGLView::Show(bool Maximized)
{
	RECT dRect, wRect, cRect;

	GetWindowRect(GetDesktopWindow(), &dRect);
	GetWindowRect(hWnd, &wRect);
	GetClientRect(hWnd, &cRect);

	wRect.right += Width - cRect.right;
	wRect.bottom += Height - cRect.bottom;
	wRect.right -= wRect.left;
	wRect.bottom -= wRect.top;
	wRect.left = dRect.right / 2 - wRect.right / 2;
	wRect.top = dRect.bottom / 2 - wRect.bottom / 2;

	MoveWindow(hWnd, wRect.left, wRect.top, wRect.right, wRect.bottom, FALSE);

	ShowWindow(hWnd, Maximized ? SW_SHOWMAXIMIZED : SW_SHOWNORMAL);
}

void COpenGLView::MessageLoop()
{
	MSG Msg;

	while(GetMessage(&Msg, NULL, 0, 0) > 0)
	{
		TranslateMessage(&Msg);
		DispatchMessage(&Msg);
	}
}

void COpenGLView::Destroy()
{
	if(GLEW_VERSION_2_1)
	{
		OpenGLRenderer.Destroy();
	}

	wglDeleteContext(hGLRC);
	DestroyWindow(hWnd);
}

void COpenGLView::OnKeyDown(UINT Key)
{
	OpenGLRenderer.OnKeyDown(Key);
}

void COpenGLView::OnLButtonDown(int X, int Y)
{
	OpenGLRenderer.OnLButtonDown(X, Y);
}

void COpenGLView::OnLButtonUp(int X, int Y)
{
	OpenGLRenderer.OnLButtonUp(X, Y);
}

void COpenGLView::OnMouseMove(int X, int Y)
{
	OpenGLRenderer.OnMouseMove(X, Y);
}

void COpenGLView::OnMouseWheel(short zDelta)
{
	OpenGLRenderer.OnMouseWheel(zDelta);
}

void COpenGLView::OnPaint()
{
	static DWORD LastFPSTime = GetTickCount(), LastFrameTime = LastFPSTime, FPS = 0;

	PAINTSTRUCT ps;

	HDC hDC = BeginPaint(hWnd, &ps);

	DWORD Time = GetTickCount();

	float FrameTime = (Time - LastFrameTime) * 0.001f;

	LastFrameTime = Time;

	if(Time - LastFPSTime > 1000)
	{
		CString Text = Title;

		if(OpenGLRenderer.Text[0] != 0)
		{
			Text.Append(" - " + OpenGLRenderer.Text);
		}

		Text.Append(" - %dx%d", Width, Height);
		Text.Append(", ATF %dx", gl_max_texture_max_anisotropy_ext);
		Text.Append(", MSAA %dx", Samples);
		Text.Append(", FPS: %d", FPS);
		Text.Append(" - %s", glGetString(GL_RENDERER));

		SetWindowText(hWnd, Text);

		LastFPSTime = Time;
		FPS = 0;
	}
	else
	{
		FPS++;
	}

	OpenGLRenderer.CheckCameraKeys(FrameTime);

	OpenGLRenderer.Render();

	OpenGLRenderer.Animate(FrameTime);

	SwapBuffers(hDC);

	EndPaint(hWnd, &ps);

	InvalidateRect(hWnd, NULL, FALSE);
}

void COpenGLView::OnRButtonDown(int X, int Y)
{
	OpenGLRenderer.OnRButtonDown(X, Y);
}

void COpenGLView::OnRButtonUp(int X, int Y)
{
	OpenGLRenderer.OnRButtonUp(X, Y);
}

void COpenGLView::OnSize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	OpenGLRenderer.Resize(Width, Height);
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView OpenGLView;

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uiMsg)
	{
		case WM_CLOSE:
			PostQuitMessage(0);
			break;

		case WM_KEYDOWN:
			OpenGLView.OnKeyDown((UINT)wParam);
			break;

		case WM_LBUTTONDOWN:
			OpenGLView.OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_LBUTTONUP:
			OpenGLView.OnLButtonUp(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_MOUSEMOVE:
			OpenGLView.OnMouseMove(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_MOUSWHEEL:
			OpenGLView.OnMouseWheel(HIWORD(wParam));
			break;

		case WM_PAINT:
			OpenGLView.OnPaint();
			break;

		case WM_RBUTTONDOWN:
			OpenGLView.OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_RBUTTONUP:
			OpenGLView.OnRButtonUp(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_SIZE:
			OpenGLView.OnSize(LOWORD(lParam), HIWORD(lParam));
			break;

		default:
			return DefWindowProc(hWnd, uiMsg, wParam, lParam);
	}

	return 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow)
{
	char *AppName = "First person camera, collision detection, gravity, jump, crouch";

	if(OpenGLView.Init(hInstance, AppName, 800, 600, 0))
	{
		OpenGLView.Show();
		OpenGLView.MessageLoop();
	}
	else
	{
		MessageBox(NULL, ErrorLog, AppName, MB_OK | MB_ICONERROR);
	}

	OpenGLView.Destroy();

	return 0;
}
