#include "object.h"

// ----------------------------------------------------------------------------------------------------------------------------

CObject::CObject()
{
	SetDefaults();
}

CObject::~CObject()
{
}

void CObject::AddData(void *Data, int DataSize)
{
	Buffer.AddData(Data, DataSize);
}

void CObject::Destroy()
{
	Texture.Destroy();

	Buffer.Empty();

	glDeleteBuffers(1, &VBO);

	delete [] Triangles;

	SetDefaults();
}

void CObject::InitVBO(int VertexOffset, int Stride)
{
	this->VertexOffset = VertexOffset;
	this->Stride = Stride;

	Buffer.FreeUnusedMemory();

	glGenBuffers(1, &VBO);
	glBindBuffer(GL_ARRAY_BUFFER, VBO);
	glBufferData(GL_ARRAY_BUFFER, Buffer.GetDataSize(), Buffer.GetData(), GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	VerticesCount = Buffer.GetDataSize() / Stride;
	TrianglesCount = VerticesCount / 3;

	Triangles = new CTriangle[TrianglesCount];

	PrepareTriangles();
}

void CObject::PrepareTriangles()
{
	BYTE *Vertices = Buffer.GetData() + VertexOffset;

	CTriangle *Triangle = Triangles;

	for(int t = 0; t < TrianglesCount; t++)
	{
		for(int v = 0; v < 3; v++)
		{
			vec3 *Vertex = (vec3*)(Vertices + Stride * v);

			Triangle->Vertex[v].x = ModelMatrix.M[0] * Vertex->x + ModelMatrix.M[4] * Vertex->y + ModelMatrix.M[8] * Vertex->z + ModelMatrix.M[12];
			Triangle->Vertex[v].y = ModelMatrix.M[1] * Vertex->x + ModelMatrix.M[5] * Vertex->y + ModelMatrix.M[9] * Vertex->z + ModelMatrix.M[13];
			Triangle->Vertex[v].z = ModelMatrix.M[2] * Vertex->x + ModelMatrix.M[6] * Vertex->y + ModelMatrix.M[10] * Vertex->z + ModelMatrix.M[14];
		}

		Triangle->Normal[0] = normalize(cross(Triangle->Vertex[1] - Triangle->Vertex[0], Triangle->Vertex[2] - Triangle->Vertex[0]));
		Triangle->D[0] = -dot(Triangle->Normal[0], Triangle->Vertex[0]);

		for(int e = 0; e < 3; e++)
		{
			Triangle->Edge[e] = Triangle->Vertex[(e + 1) % 3] - Triangle->Vertex[e];
			Triangle->EdgeLength[e] = length(Triangle->Edge[e]);
			Triangle->Edge[e] /= Triangle->EdgeLength[e];
			Triangle->Normal[1 + e] = cross(Triangle->Edge[e], Triangle->Normal[0]);
			Triangle->D[1 + e] = -dot(Triangle->Normal[1 + e], Triangle->Vertex[e]);
		}

		Vertices += Stride * 3;

		Triangle++;
	}

	Min = Max = Triangles[0].Vertex[0];

	Triangle = Triangles;

	for(int t = 0; t < TrianglesCount; t++)
	{
		for(int v = 0; v < 3; v++)
		{
			if(Triangle->Vertex[v].x < Min.x) Min.x = Triangle->Vertex[v].x;
			if(Triangle->Vertex[v].x > Max.x) Max.x = Triangle->Vertex[v].x;
			if(Triangle->Vertex[v].y < Min.y) Min.y = Triangle->Vertex[v].y;
			if(Triangle->Vertex[v].y > Max.y) Max.y = Triangle->Vertex[v].y;
			if(Triangle->Vertex[v].z < Min.z) Min.z = Triangle->Vertex[v].z;
			if(Triangle->Vertex[v].z > Max.z) Max.z = Triangle->Vertex[v].z;
		}

		Triangle++;
	}

	Min -= RADIUS;
	Max += RADIUS;
}

void CObject::SetModelMatrix(const mat4x4 &ModelMatrix)
{
	this->ModelMatrix = ModelMatrix;

	NormalMatrix = transpose(inverse(mat3x3(ModelMatrix)));

	PrepareTriangles();
}

void CObject::SetDefaults()
{
	NormalMatrix = mat3x3();
	ModelMatrix = mat4x4();

	VBO = 0;

	VertexOffset = 0;
	Stride = 0;
	VerticesCount = 0;
	TrianglesCount = 0;

	Triangles = NULL;
}
