#include "opengl_21_tutorials_win32_framework.h"

// ----------------------------------------------------------------------------------------------------------------------------

CBuffer::CBuffer()
{
	SetDefaults();
}

CBuffer::~CBuffer()
{
	Empty();
}

void CBuffer::AddData(void *Data, int DataSize)
{
	int Remaining = BufferSize - Position;

	if(DataSize > Remaining)
	{
		BYTE *OldBuffer = Buffer;
		int OldBufferSize = BufferSize;

		int Needed = DataSize - Remaining;

		BufferSize += Needed > BUFFER_SIZE_INCREMENT ? Needed : BUFFER_SIZE_INCREMENT;

		Buffer = new BYTE[BufferSize];

		memcpy(Buffer, OldBuffer, OldBufferSize);

		delete [] OldBuffer;
	}

	memcpy(Buffer + Position, Data, DataSize);

	Position += DataSize;
}

void CBuffer::Empty()
{
	delete [] Buffer;

	SetDefaults();
}

void *CBuffer::GetData()
{
	return Buffer;
}

int CBuffer::GetDataSize()
{
	return Position;
}

void CBuffer::SetDefaults()
{
	Buffer = NULL;

	BufferSize = 0;
	Position = 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int gl_max_texture_size = 0, gl_max_texture_max_anisotropy_ext = 0;

// ----------------------------------------------------------------------------------------------------------------------------

CTexture::CTexture()
{
	Texture = 0;
}

CTexture::~CTexture()
{
}

CTexture::operator GLuint ()
{
	return Texture;
}

bool CTexture::LoadTexture2D(char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + "Textures\\" + FileName;

	int Width, Height, BPP;

	FIBITMAP *dib = GetBitmap(DirectoryFileName, Width, Height, BPP);

	if(dib == NULL)
	{
		ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
		return false;
	}

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileName);
		FreeImage_Unload(dib);
		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_2D, Texture);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib));

	glBindTexture(GL_TEXTURE_2D, 0);

	FreeImage_Unload(dib);

	return true;
}

bool CTexture::LoadTextureCubeMap(char **FileNames)
{
	int Width, Height, BPP;

	FIBITMAP *dib[6];

	bool Error = false;
	
	for(int i = 0; i < 6; i++)
	{
		CString DirectoryFileName = ModuleDirectory + "Textures\\" + FileNames[i];

		dib[i] = GetBitmap(DirectoryFileName, Width, Height, BPP);

		if(dib[i] == NULL)
		{
			ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
			Error = true;
		}
	}

	if(Error)
	{
		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	GLenum Format = 0;
	
	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileNames[5]);

		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_CUBE_MAP, Texture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_GENERATE_MIPMAP, GL_TRUE);

	for(int i = 0; i < 6; i++)
	{
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib[i]));
	}

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	for(int i = 0; i < 6; i++)
	{
		FreeImage_Unload(dib[i]);
	}

	return true;
}

void CTexture::Destroy()
{
	glDeleteTextures(1, &Texture);
	Texture = 0;
}

FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP)
{
	FREE_IMAGE_FORMAT fif = FreeImage_GetFileType(FileName);

	if(fif == FIF_UNKNOWN)
	{
		fif = FreeImage_GetFIFFromFilename(FileName);
	}

	if(fif == FIF_UNKNOWN)
	{
		return NULL;
	}

	FIBITMAP *dib = NULL;

	if(FreeImage_FIFSupportsReading(fif))
	{
		dib = FreeImage_Load(fif, FileName);
	}

	if(dib != NULL)
	{
		int OriginalWidth = FreeImage_GetWidth(dib);
		int OriginalHeight = FreeImage_GetHeight(dib);

		Width = OriginalWidth;
		Height = OriginalHeight;

		if(Width == 0 || Height == 0)
		{
			FreeImage_Unload(dib);
			return NULL;
		}

		BPP = FreeImage_GetBPP(dib);

		if(Width > gl_max_texture_size) Width = gl_max_texture_size;
		if(Height > gl_max_texture_size) Height = gl_max_texture_size;

		if(!GLEW_ARB_texture_non_power_of_two)
		{
			Width = 1 << (int)floor((log((float)Width) / log(2.0f)) + 0.5f); 
			Height = 1 << (int)floor((log((float)Height) / log(2.0f)) + 0.5f);
		}

		if(Width != OriginalWidth || Height != OriginalHeight)
		{
			FIBITMAP *rdib = FreeImage_Rescale(dib, Width, Height, FILTER_BICUBIC);
			FreeImage_Unload(dib);
			dib = rdib;
		}
	}

	return dib;
}

// ----------------------------------------------------------------------------------------------------------------------------

CShaderProgram::CShaderProgram()
{
	SetDefaults();
}

CShaderProgram::~CShaderProgram()
{
}

CShaderProgram::operator GLuint ()
{
	return Program;
}

bool CShaderProgram::Load(char *VertexShaderFileName, char *FragmentShaderFileName)
{
	bool Error = false;

	Destroy();

	Error |= ((VertexShader = LoadShader(VertexShaderFileName, GL_VERTEX_SHADER)) == 0);
	Error |= ((FragmentShader = LoadShader(FragmentShaderFileName, GL_FRAGMENT_SHADER)) == 0);

	if(Error)
	{
		Destroy();
		return false;
	}

	Program = glCreateProgram();
	glAttachShader(Program, VertexShader);
	glAttachShader(Program, FragmentShader);
	glLinkProgram(Program);

	int LinkStatus;
	glGetProgramiv(Program, GL_LINK_STATUS, &LinkStatus);

	if(LinkStatus == GL_FALSE)
	{
		ErrorLog.Append("Error linking program (%s, %s)!\r\n", VertexShaderFileName, FragmentShaderFileName);

		int InfoLogLength = 0;
		glGetProgramiv(Program, GL_INFO_LOG_LENGTH, &InfoLogLength);
	
		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetProgramInfoLog(Program, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		Destroy();

		return false;
	}

	return true;
}

void CShaderProgram::Destroy()
{
	glDetachShader(Program, VertexShader);
	glDetachShader(Program, FragmentShader);

	glDeleteShader(VertexShader);
	glDeleteShader(FragmentShader);

	glDeleteProgram(Program);

	delete [] UniformLocations;
	delete [] AttribLocations;

	SetDefaults();
}

GLuint CShaderProgram::LoadShader(char *FileName, GLenum Type)
{
	CString DirectoryFileName = ModuleDirectory + "Shaders\\" + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file " + DirectoryFileName + "!\r\n");
		return 0;
	}

	fseek(File, 0, SEEK_END);
	long Size = ftell(File);
	fseek(File, 0, SEEK_SET);
	char *Source = new char[Size + 1];
	fread(Source, 1, Size, File);
	fclose(File);
	Source[Size] = 0;

	GLuint Shader = glCreateShader(Type);

	glShaderSource(Shader, 1, (const char**)&Source, NULL);
	delete [] Source;
	glCompileShader(Shader);

	int CompileStatus;
	glGetShaderiv(Shader, GL_COMPILE_STATUS, &CompileStatus);

	if(CompileStatus == GL_FALSE)
	{
		ErrorLog.Append("Error compiling shader %s!\r\n", FileName);

		int InfoLogLength = 0;
		glGetShaderiv(Shader, GL_INFO_LOG_LENGTH, &InfoLogLength);
	
		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetShaderInfoLog(Shader, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		glDeleteShader(Shader);

		return 0;
	}

	return Shader;
}

void CShaderProgram::SetDefaults()
{
	VertexShader = 0;
	FragmentShader = 0;

	Program = 0;

	UniformLocations = NULL;
	AttribLocations = NULL;
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera::CCamera()
{
	ViewMatrix = NULL;
	ViewMatrixInverse = NULL;

	X = vec3(1.0f, 0.0f, 0.0f);
	Y = vec3(0.0f, 1.0f, 0.0f);
	Z = vec3(0.0f, 0.0f, 1.0f);

	Position = vec3(0.0f, 0.0f, 5.0f);
	Reference = vec3(0.0f, 0.0f, 0.0f);
}

CCamera::~CCamera()
{
}

void CCamera::Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference)
{
	this->Position = Position;
	this->Reference = Reference;

	Z = normalize(Position - Reference);
	X = normalize(cross(vec3(0.0f, 1.0f, 0.0f), Z));
	Y = cross(Z, X);

	if(!RotateAroundReference)
	{
		this->Reference = this->Position;
		this->Position += Z * 0.05f;
	}

	CalculateViewMatrix();
}

void CCamera::Move(const vec3 &Movement)
{
	Position += Movement;
	Reference += Movement;

	CalculateViewMatrix();
}

vec3 CCamera::OnKeys(BYTE Keys, float FrameTime)
{
	float Speed = 5.0f;

	if(Keys & 0x40) Speed *= 2.0f;
	if(Keys & 0x80) Speed *= 0.5f;

	float Distance = Speed * FrameTime;

	vec3 Up(0.0f, 1.0f, 0.0f);
	vec3 Right = X;
	vec3 Forward = cross(Up, Right);

	Up *= Distance;
	Right *= Distance;
	Forward *= Distance;

	vec3 Movement;

	if(Keys & 0x01) Movement += Forward;
	if(Keys & 0x02) Movement -= Forward;
	if(Keys & 0x04) Movement -= Right;
	if(Keys & 0x08) Movement += Right;
	if(Keys & 0x10) Movement += Up;
	if(Keys & 0x20) Movement -= Up;

	return Movement;
}

void CCamera::OnMouseMove(int dx, int dy)
{
	float Sensitivity = 0.25f;

	Position -= Reference;

	if(dx != 0)
	{
		float DeltaX = (float)dx * Sensitivity;

		X = rotate(X, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Y = rotate(Y, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Z = rotate(Z, DeltaX, vec3(0.0f, 1.0f, 0.0f));
	}

	if(dy != 0)
	{
		float DeltaY = (float)dy * Sensitivity;

		Y = rotate(Y, DeltaY, X);
		Z = rotate(Z, DeltaY, X);

		if(Y.y < 0.0f)
		{
			Z = vec3(0.0f, Z.y > 0.0f ? 1.0f : -1.0f, 0.0f);
			Y = cross(Z, X);
		}
	}

	Position = Reference + Z * length(Position);

	CalculateViewMatrix();
}

void CCamera::OnMouseWheel(float zDelta)
{
	Position -= Reference;

	if(zDelta < 0 && length(Position) < 500.0f)
	{
		Position += Position * 0.1f;
	}

	if(zDelta > 0 && length(Position) > 0.05f)
	{
		Position -= Position * 0.1f;
	}

	Position += Reference;

	CalculateViewMatrix();
}

void CCamera::SetViewMatrixPointer(float *ViewMatrix, float *ViewMatrixInverse)
{
	this->ViewMatrix = (mat4x4*)ViewMatrix;
	this->ViewMatrixInverse = (mat4x4*)ViewMatrixInverse;

	CalculateViewMatrix();
}

void CCamera::CalculateViewMatrix()
{
	if(ViewMatrix != NULL)
	{
		*ViewMatrix = mat4x4(X.x, Y.x, Z.x, 0.0f, X.y, Y.y, Z.y, 0.0f, X.z, Y.z, Z.z, 0.0f, -dot(X, Position), -dot(Y, Position), -dot(Z, Position), 1.0f);

		if(ViewMatrixInverse != NULL)
		{
			*ViewMatrixInverse = inverse(*ViewMatrix);
		}
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera Camera;

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer::COpenGLRenderer()
{
	Texturing = true;
	NormalMapping = true;
	CalculateShadows = true;
	DoShadowFiltering = true;
	CalculateSSAO = true;
	ShowSSAO = false;
	BlurSSAO = true;
	FullSSAO = false;
	CalculateAntialiasing = true;

	Pause = false;

	RenderGLUTObjects = true;

	Camera.SetViewMatrixPointer(&ViewMatrix, &ViewMatrixInverse);
}

COpenGLRenderer::~COpenGLRenderer()
{
}

bool COpenGLRenderer::Init()
{
	bool Error = false;

	// ------------------------------------------------------------------------------------------------------------------------

	if(!GLEW_ARB_texture_non_power_of_two)
	{
		ErrorLog.Append("GL_ARB_texture_non_power_of_two not supported!\r\n");
		Error = true;
	}

	if(!GLEW_ARB_depth_texture)
	{
		ErrorLog.Append("GL_ARB_depth_texture not supported!\r\n");
		Error = true;
	}

	if(!GLEW_EXT_framebuffer_object)
	{
		ErrorLog.Append("GL_EXT_framebuffer_object not supported!\r\n");
		Error = true;
	}

	// ------------------------------------------------------------------------------------------------------------------------

	Error |= !Texture[0].LoadTexture2D("texture2.jpg");
	Error |= !Texture[1].LoadTexture2D("metal2.jpg");
	Error |= !Texture[2].LoadTexture2D("colormap1.jpg");
	Error |= !Texture[3].LoadTexture2D("normalmap1.jpg");

	// ------------------------------------------------------------------------------------------------------------------------

	Error |= !InitScene();

	// ------------------------------------------------------------------------------------------------------------------------

	Error |= !Preprocess.Load("preprocess.vs", "preprocess.fs");
	Error |= !SSAO.Load("ssao.vs", "ssao.fs");
	Error |= !SSAOFilterH.Load("ssaofilter.vs", "ssaofilterh.fs");
	Error |= !SSAOFilterV.Load("ssaofilter.vs", "ssaofilterv.fs");
	Error |= !DeferredLighting.Load("deferredlighting.vs", "deferredlighting.fs");
	Error |= !Antialiasing.Load("antialiasing.vs", "antialiasing.fs");

	// ------------------------------------------------------------------------------------------------------------------------

	if(Error)
	{
		return false;
	}

	// ------------------------------------------------------------------------------------------------------------------------

	Preprocess.AttribLocations = new GLuint[1];
	Preprocess.AttribLocations[0] = glGetAttribLocation(Preprocess, "vert_Tangent");

	// ------------------------------------------------------------------------------------------------------------------------

	SSAO.UniformLocations = new GLuint[2];
	SSAO.UniformLocations[0] = glGetUniformLocation(SSAO, "Scale");
	SSAO.UniformLocations[1] = glGetUniformLocation(SSAO, "ProjectionBiasMatrixInverse");

	SSAOFilterH.UniformLocations = new GLuint[1];
	SSAOFilterH.UniformLocations[0] = glGetUniformLocation(SSAOFilterH, "PixelSizeX");

	SSAOFilterV.UniformLocations = new GLuint[1];
	SSAOFilterV.UniformLocations[0] = glGetUniformLocation(SSAOFilterV, "PixelSizeY");

	DeferredLighting.UniformLocations = new GLuint[8];
	DeferredLighting.UniformLocations[0] = glGetUniformLocation(DeferredLighting, "CalculateShadows");
	DeferredLighting.UniformLocations[1] = glGetUniformLocation(DeferredLighting, "DoShadowFiltering");
	DeferredLighting.UniformLocations[2] = glGetUniformLocation(DeferredLighting, "CalculateSSAO");
	DeferredLighting.UniformLocations[3] = glGetUniformLocation(DeferredLighting, "LightPosition");
	DeferredLighting.UniformLocations[4] = glGetUniformLocation(DeferredLighting, "LightNormal");
	DeferredLighting.UniformLocations[5] = glGetUniformLocation(DeferredLighting, "ShadowMatrix");
	DeferredLighting.UniformLocations[6] = glGetUniformLocation(DeferredLighting, "Scale");
	DeferredLighting.UniformLocations[7] = glGetUniformLocation(DeferredLighting, "ProjectionBiasMatrixInverse");

	Antialiasing.UniformLocations = new GLuint[1];
	Antialiasing.UniformLocations[0] = glGetUniformLocation(Antialiasing, "PixelSize");

	// ------------------------------------------------------------------------------------------------------------------------

	glUseProgram(Preprocess);
	glUniform1i(glGetUniformLocation(Preprocess, "Texture"), 0);
	glUniform1i(glGetUniformLocation(Preprocess, "NormalMap"), 1);
	glUseProgram(0);

	glUseProgram(SSAO);
	glUniform1i(glGetUniformLocation(SSAO, "NormalBuffer"), 0);
	glUniform1i(glGetUniformLocation(SSAO, "DepthBuffer"), 1);
	glUniform1i(glGetUniformLocation(SSAO, "RotationTexture"), 2);
	glUseProgram(0);

	float s = 128.0f, e = 131070.0f, fs = 1.0f / s, fe = 1.0f / e, fd = fs - fe;

	glUseProgram(SSAOFilterH);
	glUniform1i(glGetUniformLocation(SSAOFilterH, "SSAOBuffer"), 0);
	glUniform1i(glGetUniformLocation(SSAOFilterH, "DepthBuffer"), 1);
	glUniform1f(glGetUniformLocation(SSAOFilterH, "fs"), fs);
	glUniform1f(glGetUniformLocation(SSAOFilterH, "fd"), fd);
	glUseProgram(0);

	glUseProgram(SSAOFilterV);
	glUniform1i(glGetUniformLocation(SSAOFilterV, "SSAOBuffer"), 0);
	glUniform1i(glGetUniformLocation(SSAOFilterV, "DepthBuffer"), 1);
	glUniform1f(glGetUniformLocation(SSAOFilterV, "fs"), fs);
	glUniform1f(glGetUniformLocation(SSAOFilterV, "fd"), fd);
	glUseProgram(0);

	glUseProgram(DeferredLighting);
	glUniform1i(glGetUniformLocation(DeferredLighting, "ColorBuffer"), 0);
	glUniform1i(glGetUniformLocation(DeferredLighting, "NormalBuffer"), 1);
	glUniform1i(glGetUniformLocation(DeferredLighting, "DepthBuffer"), 2);
	glUniform1i(glGetUniformLocation(DeferredLighting, "SSAOBuffer"), 3);
	glUniform1i(glGetUniformLocation(DeferredLighting, "RotationTexture"), 4);
	glUniform1i(glGetUniformLocation(DeferredLighting, "ShadowMap"), 5);
	glUseProgram(0);

	glUseProgram(Antialiasing);
	glUniform1i(glGetUniformLocation(Antialiasing, "ColorBuffer"), 0);
	glUniform1i(glGetUniformLocation(Antialiasing, "NormalBuffer"), 1);
	glUniform1i(glGetUniformLocation(Antialiasing, "DepthBuffer"), 2);
	glUseProgram(0);

	// ------------------------------------------------------------------------------------------------------------------------

	vec2 *Samples = new vec2[16];

	float Angle = 0.0f, Radius = 0.416666;

	for(int i = 0; i < 16; i++)
	{
		Samples[i].x = sin(Angle) * ((float)i + 1.0f) / 16.0f * Radius;
		Samples[i].y = cos(Angle) * ((float)i + 1.0f) / 16.0f * Radius;

		Angle += M_PI_2;

		if(!((i + 1) % 4)) Angle += M_PI_4;
	}

	glUseProgram(SSAO);
	glUniform2fv(glGetUniformLocation(SSAO, "Samples"), 16, (float*)Samples);
	glUseProgram(0);

	// ------------------------------------------------------------------------------------------------------------------------

	Angle = 0.0f;

	for(int i = 0; i < 16; i++)
	{
		Samples[i].x = sin(Angle) * ((float)i + 1.0f) / 16.0f / 1024.0f * 2.0f;
		Samples[i].y = cos(Angle) * ((float)i + 1.0f) / 16.0f / 1024.0f * 2.0f;

		Angle += M_PI_2;

		if(!((i + 1) % 4)) Angle += M_PI_4;
	}

	glUseProgram(DeferredLighting);
	glUniform2fv(glGetUniformLocation(DeferredLighting, "Samples"), 16, (float*)Samples);
	glUseProgram(0);

	delete [] Samples;

	// ------------------------------------------------------------------------------------------------------------------------

	vec4 *RotationTextureData = new vec4[64 * 64];

	float RandomAngle = 0.0f;

	for(int i = 0; i < 64 * 64; i++)
	{
		RotationTextureData[i].x = cos(RandomAngle) * 0.5 + 0.5;
		RotationTextureData[i].y = sin(RandomAngle) * 0.5 + 0.5;
		RotationTextureData[i].z = -sin(RandomAngle) * 0.5 + 0.5;
		RotationTextureData[i].w = cos(RandomAngle) * 0.5 + 0.5;

		RandomAngle += (float)rand() / (float)RAND_MAX * M_PI * 2.0;
	}

	glGenTextures(1, &RotationTexture);
    glBindTexture(GL_TEXTURE_2D, RotationTexture);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, 64, 64, 0, GL_RGBA, GL_FLOAT, RotationTextureData);
	glBindTexture(GL_TEXTURE_2D, 0);

	delete [] RotationTextureData;

	// ------------------------------------------------------------------------------------------------------------------------

	glGenTextures(1, &ShadowMap);
	glBindTexture(GL_TEXTURE_2D, ShadowMap);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE, GL_COMPARE_R_TO_TEXTURE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC, GL_LEQUAL);
	glTexParameteri(GL_TEXTURE_2D, GL_DEPTH_TEXTURE_MODE, GL_INTENSITY);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT24, 1024, 1024, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	// ------------------------------------------------------------------------------------------------------------------------

	glGenTextures(2, ColorBuffers);
	glGenTextures(1, &NormalBuffer);
	glGenTextures(1, &DepthBuffer);
	glGenTextures(2, SSAOBuffers);

	// ------------------------------------------------------------------------------------------------------------------------

	glGenFramebuffersEXT(1, &FBO);

	// ------------------------------------------------------------------------------------------------------------------------

	Camera.Look(vec3(-1.0f, 1.75, 1.0f), vec3(0.0f, 1.75, 0.0f));

	return true;
}

void COpenGLRenderer::Render(float FrameTime)
{
	GLenum Buffers[2] = {GL_COLOR_ATTACHMENT0_EXT, GL_COLOR_ATTACHMENT1_EXT};

	// ------------------------------------------------------------------------------------------------------------------------

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
	glDrawBuffers(2, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, ColorBuffers[0], 0);
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, NormalBuffer, 0);
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, DepthBuffer, 0);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&ProjectionMatrix);

	glUseProgram(Preprocess);

	RenderScene();

	glUseProgram(0);

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	// ------------------------------------------------------------------------------------------------------------------------

	static vec3 LightPosition = vec3(1.0f, 3.875f, 0.0f);
	static mat4x4 LightProjectionMatrix = perspective(139.0f, 1.0f, 0.125f, 512.0f);

	// ------------------------------------------------------------------------------------------------------------------------

	if(!Pause)
	{
		LightPosition = rotate(LightPosition, 90.0f * FrameTime, vec3(0.0f, 1.0f, 0.0f));
	}

	// ------------------------------------------------------------------------------------------------------------------------

	vec4 CameraSpaceLightPosition = ViewMatrix * vec4(LightPosition, 1.0);
	vec3 CameraSpaceLightNormal = vec3(-ViewMatrix[4], -ViewMatrix[5], -ViewMatrix[6]);

	// ------------------------------------------------------------------------------------------------------------------------

	mat4x4 LightViewMatrix = look(LightPosition, LightPosition - vec3(0.0f, 1.0f, 0.0f), vec3(0.0f, 0.0f, -1.0f));
	mat4x4 ShadowMatrix = BiasMatrix * LightProjectionMatrix * LightViewMatrix * ViewMatrixInverse;

	if(CalculateShadows && !Pause && !ShowSSAO)
	{
		glViewport(0, 0, 1024, 1024);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glDrawBuffers(0, NULL); glReadBuffer(GL_NONE);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, 0, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, ShadowMap, 0);

		glClear(GL_DEPTH_BUFFER_BIT);

		glMatrixMode(GL_PROJECTION);
		glLoadMatrixf(&LightProjectionMatrix);

		RenderSceneToShadowMap(LightViewMatrix);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		glViewport(0, 0, Width, Height);
	}

	// ------------------------------------------------------------------------------------------------------------------------

	if(CalculateSSAO || ShowSSAO)
	{
		if(!FullSSAO)
		{
			glViewport(0, 0, Width / 2, Height / 2);
		}

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glDrawBuffers(1, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, SSAOBuffers[0], 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, NormalBuffer);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, DepthBuffer);
		glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, RotationTexture);

		glUseProgram(SSAO);

		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();

		glUseProgram(0);

		glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		// --------------------------------------------------------------------------------------------------------------------

		if(BlurSSAO)
		{
			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
			glDrawBuffers(1, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, SSAOBuffers[1], 0);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

			glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, SSAOBuffers[0]);
			glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, DepthBuffer);

			glUseProgram(SSAOFilterH);

			glBegin(GL_QUADS);
				glVertex2f(0.0f, 0.0f);
				glVertex2f(1.0f, 0.0f);
				glVertex2f(1.0f, 1.0f);
				glVertex2f(0.0f, 1.0f);
			glEnd();

			glUseProgram(0);

			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, SSAOBuffers[0], 0);

			glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, SSAOBuffers[1]);
			glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, DepthBuffer);

			glUseProgram(SSAOFilterV);

			glBegin(GL_QUADS);
				glVertex2f(0.0f, 0.0f);
				glVertex2f(1.0f, 0.0f);
				glVertex2f(1.0f, 1.0f);
				glVertex2f(0.0f, 1.0f);
			glEnd();

			glUseProgram(0);

			glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
			glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}

		if(!FullSSAO)
		{
			glViewport(0, 0, Width, Height);
		}
	}

	// ------------------------------------------------------------------------------------------------------------------------

	if(ShowSSAO)
	{
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		glEnable(GL_TEXTURE_2D);

		glBindTexture(GL_TEXTURE_2D, SSAOBuffers[0]);

		glBegin(GL_QUADS);
			glTexCoord2f(0.0f, 0.0f); glVertex2f(-1.0f, -1.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex2f(1.0f, -1.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex2f(1.0f, 1.0f);
			glTexCoord2f(0.0f, 1.0f); glVertex2f(-1.0f, 1.0f);
		glEnd();

		glBindTexture(GL_TEXTURE_2D, 0);

		glDisable(GL_TEXTURE_2D);
	}
	else
	{
		if(CalculateAntialiasing)
		{
			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
			glDrawBuffers(1, Buffers); glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, ColorBuffers[1], 0);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);
		}

		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, ColorBuffers[0]);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, NormalBuffer);
		glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, DepthBuffer);
		glActiveTexture(GL_TEXTURE3); glBindTexture(GL_TEXTURE_2D, SSAOBuffers[0]);
		glActiveTexture(GL_TEXTURE4); glBindTexture(GL_TEXTURE_2D, RotationTexture);
		glActiveTexture(GL_TEXTURE5); glBindTexture(GL_TEXTURE_2D, ShadowMap);

		glUseProgram(DeferredLighting);
		glUniform1i(DeferredLighting.UniformLocations[0], CalculateShadows);
		glUniform1i(DeferredLighting.UniformLocations[1], DoShadowFiltering);
		glUniform1i(DeferredLighting.UniformLocations[2], CalculateSSAO);
		glUniform3fv(DeferredLighting.UniformLocations[3], 1, &CameraSpaceLightPosition);
		glUniform3fv(DeferredLighting.UniformLocations[4], 1, &CameraSpaceLightNormal);
		glUniformMatrix4fv(DeferredLighting.UniformLocations[5], 1, GL_FALSE, &ShadowMatrix);

		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();

		glUseProgram(0);

		glActiveTexture(GL_TEXTURE5); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE4); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE3); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

		if(CalculateAntialiasing)
		{
			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}

		// --------------------------------------------------------------------------------------------------------------------------

		if(CalculateAntialiasing)
		{
			glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, ColorBuffers[1]);
			glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, NormalBuffer);
			glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, DepthBuffer);
			glUseProgram(Antialiasing);
			glBegin(GL_QUADS);
				glVertex2f(0.0f, 0.0f);
				glVertex2f(1.0f, 0.0f);
				glVertex2f(1.0f, 1.0f);
				glVertex2f(0.0f, 1.0f);
			glEnd();
			glUseProgram(0);
			glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, 0);
			glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
			glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);
		}
	}
}

void COpenGLRenderer::Resize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	glViewport(0, 0, Width, Height);

	ProjectionMatrix = perspective(45.0f, (float)Width / (float)Height, 0.125f, 512.0f);
	ProjectionBiasMatrixInverse = inverse(ProjectionMatrix) * BiasMatrixInverse;                      

	for(int i = 0; i < 2; i++)
	{
		glBindTexture(GL_TEXTURE_2D, ColorBuffers[i]);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
		glBindTexture(GL_TEXTURE_2D, 0);
	}

	glBindTexture(GL_TEXTURE_2D, NormalBuffer);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	glBindTexture(GL_TEXTURE_2D, DepthBuffer);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT24, Width, Height, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL);
	glBindTexture(GL_TEXTURE_2D, 0);

	for(int i = 0; i < 2; i++)
	{
		glBindTexture(GL_TEXTURE_2D, SSAOBuffers[i]);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, FullSSAO ? GL_NEAREST : GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, FullSSAO ? GL_NEAREST : GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, FullSSAO ? GL_CLAMP : GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, FullSSAO ? GL_CLAMP : GL_CLAMP_TO_EDGE);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width / (FullSSAO ? 1 : 2), Height / (FullSSAO ? 1 : 2), 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
		glBindTexture(GL_TEXTURE_2D, 0);
	}

	glUseProgram(SSAO);
	glUniform2f(SSAO.UniformLocations[0], (float)(Width / (FullSSAO ? 1 : 2)) / 64.0f, (float)(Height / (FullSSAO ? 1 : 2)) / 64.0f);
	glUniformMatrix4fv(SSAO.UniformLocations[1], 1, GL_FALSE, &ProjectionBiasMatrixInverse);
	glUseProgram(0);

	glUseProgram(SSAOFilterH);
	glUniform1f(SSAOFilterH.UniformLocations[0], 1.0f / (float)(Width / (FullSSAO ? 1 : 2)));
	glUseProgram(0);

	glUseProgram(SSAOFilterV);
	glUniform1f(SSAOFilterV.UniformLocations[0], 1.0f / (float)(Height / (FullSSAO ? 1 : 2)));
	glUseProgram(0);

	glUseProgram(DeferredLighting);
	glUniform2f(DeferredLighting.UniformLocations[6], (float)Width / 64.0f, (float)Height / 64.0f);
	glUniformMatrix4fv(DeferredLighting.UniformLocations[7], 1, GL_FALSE, &ProjectionBiasMatrixInverse);
	glUseProgram(0);

	glUseProgram(Antialiasing);
	glUniform2f(Antialiasing.UniformLocations[0], 1.0f / (float)Width, 1.0f / (float)Height);
	glUseProgram(0);
}

void COpenGLRenderer::Destroy()
{
	for(int i = 0; i <4; i++)
	{
		Texture[i].Destroy();
	}

	glDeleteBuffers(1, &VBO);

	Preprocess.Destroy();
	SSAO.Destroy();
	SSAOFilterH.Destroy();
	SSAOFilterV.Destroy();
	DeferredLighting.Destroy();
	Antialiasing.Destroy();

	glDeleteTextures(1, &RotationTexture);
	glDeleteTextures(1, &ShadowMap);
	glDeleteTextures(2, ColorBuffers);
	glDeleteTextures(1, &NormalBuffer);
	glDeleteTextures(1, &DepthBuffer);
	glDeleteTextures(2, SSAOBuffers);

	if(GLEW_EXT_framebuffer_object)
	{
		glDeleteFramebuffersEXT(1, &FBO);
	}
}

bool COpenGLRenderer::InitScene()
{
	CString FileName = ModuleDirectory + "Models\\" + "room.xyz";

	FILE *File;

	if(fopen_s(&File, FileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file room.xyz!\r\n");
		return false;
	}

	QuadsCount = 0;

	fread(&QuadsCount, sizeof(int), 1, File);

	CQuad *Quads = new CQuad[QuadsCount + 18];

	for(int i = 0; i < QuadsCount; i++)
	{
		fread(&Quads[i].a.x, sizeof(float), 1, File);
		fread(&Quads[i].a.y, sizeof(float), 1, File);
		fread(&Quads[i].a.z, sizeof(float), 1, File);
		fread(&Quads[i].b.x, sizeof(float), 1, File);
		fread(&Quads[i].b.y, sizeof(float), 1, File);
		fread(&Quads[i].b.z, sizeof(float), 1, File);
		fread(&Quads[i].c.x, sizeof(float), 1, File);
		fread(&Quads[i].c.y, sizeof(float), 1, File);
		fread(&Quads[i].c.z, sizeof(float), 1, File);
		fread(&Quads[i].d.x, sizeof(float), 1, File);
		fread(&Quads[i].d.y, sizeof(float), 1, File);
		fread(&Quads[i].d.z, sizeof(float), 1, File);
		fread(&Quads[i].normal.x, sizeof(float), 1, File);
		fread(&Quads[i].normal.y, sizeof(float), 1, File);
		fread(&Quads[i].normal.z, sizeof(float), 1, File);
	}

	fclose(File);

	vec3 offset = vec3(0.0f, 0.5f, 3.375f);

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f,  0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3(-0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3( 0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3( 0.5f,  0.5f,  0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3(-0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3(-0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f, -0.5f,  0.5f) + offset;

	QuadsCount++;

	offset = vec3(1.5f, 0.5f, 3.375f);

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f,  0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3(-0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3( 0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3( 0.5f,  0.5f,  0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3(-0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3(-0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f, -0.5f,  0.5f) + offset;

	QuadsCount++;

	offset = vec3(0.75f, 1.5f, 3.375f);

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f,  0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3(-0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3( 0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3( 0.5f,  0.5f,  0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3(-0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3(-0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f,  0.5f,  0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f,  0.5f, -0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f,  0.5f, -0.5f) + offset;

	QuadsCount++;

	Quads[QuadsCount].a = vec3(-0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].b = vec3( 0.5f, -0.5f, -0.5f) + offset;
	Quads[QuadsCount].c = vec3( 0.5f, -0.5f,  0.5f) + offset;
	Quads[QuadsCount].d = vec3(-0.5f, -0.5f,  0.5f) + offset;

	QuadsCount++;

	CBuffer buffer;

	for(int i = 0; i < QuadsCount; i++)
	{
		vec3 t = normalize(Quads[i].b - Quads[i].a);
		vec3 b = normalize(Quads[i].c - Quads[i].a);
		vec3 n = normalize(cross(t, b));

		b = cross(n, t);

		buffer.AddData(&vec2(dot(t, Quads[i].a), dot(b, Quads[i].a)), 8);
		buffer.AddData(&n, 12);
		buffer.AddData(&t, 12);
		buffer.AddData(&Quads[i].a, 12);

		buffer.AddData(&vec2(dot(t, Quads[i].b), dot(b, Quads[i].b)), 8);
		buffer.AddData(&n, 12);
		buffer.AddData(&t, 12);
		buffer.AddData(&Quads[i].b, 12);

		buffer.AddData(&vec2(dot(t, Quads[i].c), dot(b, Quads[i].c)), 8);
		buffer.AddData(&n, 12);
		buffer.AddData(&t, 12);
		buffer.AddData(&Quads[i].c, 12);

		buffer.AddData(&vec2(dot(t, Quads[i].d), dot(b, Quads[i].d)), 8);
		buffer.AddData(&n, 12);
		buffer.AddData(&t, 12);
		buffer.AddData(&Quads[i].d, 12);
	}

	glGenBuffers(1, &VBO);
	glBindBuffer(GL_ARRAY_BUFFER, VBO);
	glBufferData(GL_ARRAY_BUFFER, buffer.GetDataSize(), buffer.GetData(), GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	buffer.Empty();
	
	delete [] Quads;

	return true;
}

void COpenGLRenderer::RenderScene()
{
	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&ViewMatrix);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);

	glBindBuffer(GL_ARRAY_BUFFER, VBO);

	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2, GL_FLOAT, 44, (void*)0);

	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(GL_FLOAT, 44, (void*)8);

	glEnableVertexAttribArray(Preprocess.AttribLocations[0]);
	glVertexAttribPointer(Preprocess.AttribLocations[0], 3, GL_FLOAT, GL_FALSE, 44, (void*)20);

	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 44, (void*)32);

	glUniform1i(glGetUniformLocation(Preprocess, "Texturing"), Texturing);
	glUniform1i(glGetUniformLocation(Preprocess, "NormalMapping"), NormalMapping);

	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, Texture[2]);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, Texture[3]);

	glDrawArrays(GL_QUADS, 0, 24);
	glDrawArrays(GL_QUADS, QuadsCount * 4 - 24 * 3, 24 * 3);

	glUniform1i(glGetUniformLocation(Preprocess, "NormalMapping"), false);

	glColor4f(1.0f, 1.0f, 1.0f, 0.0f);
	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, Texture[0]);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
	glDrawArrays(GL_QUADS, 24, QuadsCount * 4 - 24 * 4);

	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableVertexAttribArray(Preprocess.AttribLocations[0]);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	glBindBuffer(GL_ARRAY_BUFFER, 0);

	glUniform1i(glGetUniformLocation(Preprocess, "Texturing"), false);

	glColor4f(0.0f, 0.5f, 1.0f, 1.0f);
	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);

	if(RenderGLUTObjects)
	{
		glLoadMatrixf(&ViewMatrix);
		glTranslatef(-2.5f, 0.25f, -1.0f);
		glRotatef(90, 1, 0, 0);
		glutSolidTorus(0.25f, 0.5f, 32, 64);

		glLoadMatrixf(&ViewMatrix);
		glTranslatef(-2.5f, 0.25f, 0.0f);
		glutSolidSphere(0.25f, 32, 32);

		glLoadMatrixf(&ViewMatrix);
		glTranslatef(-2.5f, 0.125f, 0.365f);
		glutSolidSphere(0.125f, 32, 32);
	}

	glUniform1i(glGetUniformLocation(Preprocess, "Texturing"), Texturing);

	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, Texture[1]);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);

	glDisable(GL_CULL_FACE);

	if(RenderGLUTObjects)
	{
		glLoadMatrixf(&ViewMatrix);
		glTranslatef(2.5f, 1.195f, -2.0f);
		glRotatef(33, 0, 1, 0);
		glutSolidTeapot(0.25f);

		glLoadMatrixf(&ViewMatrix);
		glTranslatef(2.5f, 1.195f, -2.5f);
		glRotatef(180, 0, 1, 0);
		glutSolidTeapot(0.25f);

		glLoadMatrixf(&ViewMatrix);
		glTranslatef(2.5f, 1.195f, -3.0f);
		glRotatef(-33, 0, 1, 0);
		glutSolidTeapot(0.25f);
	}

	glDisable(GL_DEPTH_TEST);
}

void COpenGLRenderer::RenderSceneToShadowMap(mat4x4 &LightViewMatrix)
{
	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&LightViewMatrix);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);

	glBindBuffer(GL_ARRAY_BUFFER, VBO);
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 44, (void*)32);
	glDrawArrays(GL_QUADS, 0, QuadsCount * 4);
	glDisableClientState(GL_VERTEX_ARRAY);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	if(RenderGLUTObjects)
	{
		glLoadMatrixf(&LightViewMatrix);
		glTranslatef(-2.5f, 0.25f, -1.0f);
		glRotatef(90, 1, 0, 0);
		glutSolidTorus(0.25f, 0.5f, 64, 64);

		glLoadMatrixf(&LightViewMatrix);
		glTranslatef(-2.5f, 0.25f, 0.0f);
		glutSolidSphere(0.25f, 32, 32);

		glLoadMatrixf(&LightViewMatrix);
		glTranslatef(-2.5f, 0.125f, 0.365f);
		glutSolidSphere(0.125f, 32, 32);
	}

	glDisable(GL_CULL_FACE);

	if(RenderGLUTObjects)
	{
		glLoadMatrixf(&LightViewMatrix);
		glTranslatef(2.5f, 1.195f, -2.0f);
		glRotatef(33, 0, 1, 0);
		glutSolidTeapot(0.25f);

		glLoadMatrixf(&LightViewMatrix);
		glTranslatef(2.5f, 1.195f, -2.5f);
		glRotatef(180, 0, 1, 0);
		glutSolidTeapot(0.25f);

		glLoadMatrixf(&LightViewMatrix);
		glTranslatef(2.5f, 1.195f, -3.0f);
		glRotatef(-33, 0, 1, 0);
		glutSolidTeapot(0.25f);
	}

	glDisable(GL_DEPTH_TEST);
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer OpenGLRenderer;

// ----------------------------------------------------------------------------------------------------------------------------

CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

void GetModuleDirectory()
{
	char *moduledirectory = new char[256];
	GetModuleFileName(GetModuleHandle(NULL), moduledirectory, 256);
	*(strrchr(moduledirectory, '\\') + 1) = 0;
	ModuleDirectory = moduledirectory;
	delete [] moduledirectory;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView::COpenGLView()
{
}

COpenGLView::~COpenGLView()
{
}

bool COpenGLView::Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples)
{
	this->Title = Title;
	this->Width = Width;
	this->Height = Height;

	WNDCLASSEX WndClassEx;

	memset(&WndClassEx, 0, sizeof(WNDCLASSEX));

	WndClassEx.cbSize = sizeof(WNDCLASSEX);
	WndClassEx.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	WndClassEx.lpfnWndProc = WndProc;
	WndClassEx.hInstance = hInstance;
	WndClassEx.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hCursor = LoadCursor(NULL, IDC_ARROW);
	WndClassEx.lpszClassName = "Win32OpenGLWindowClass";

	if(RegisterClassEx(&WndClassEx) == 0)
	{
		ErrorLog.Set("RegisterClassEx failed!");
		return false;
	}

	DWORD Style = WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	hWnd = CreateWindowEx(WS_EX_APPWINDOW, WndClassEx.lpszClassName, Title, Style, 0, 0, Width, Height, NULL, NULL, hInstance, NULL);

	if(hWnd == NULL)
	{
		ErrorLog.Set("CreateWindowEx failed!");
		return false;
	}

	HDC hDC = GetDC(hWnd);

	if(hDC == NULL)
	{
		ErrorLog.Set("GetDC failed!");
		return false;
	}

	PIXELFORMATDESCRIPTOR pfd;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 32;
	pfd.cDepthBits = 24;
	pfd.iLayerType = PFD_MAIN_PLANE;

	int PixelFormat = ChoosePixelFormat(hDC, &pfd);

	if(PixelFormat == 0)
	{
		ErrorLog.Set("ChoosePixelFormat failed!");
		return false;
	}

	static int MSAAPixelFormat = 0;

	if(SetPixelFormat(hDC, MSAAPixelFormat == 0 ? PixelFormat : MSAAPixelFormat, &pfd) == FALSE)
	{
		ErrorLog.Set("SetPixelFormat failed!");
		return false;
	}

	hGLRC = wglCreateContext(hDC);

	if(hGLRC == NULL)
	{
		ErrorLog.Set("wglCreateContext failed!");
		return false;
	}

	if(wglMakeCurrent(hDC, hGLRC) == FALSE)
	{
		ErrorLog.Set("wglMakeCurrent failed!");
		return false;
	}

	if(glewInit() != GLEW_OK)
	{
		ErrorLog.Set("glewInit failed!");
		return false;
	}

	if(!GLEW_VERSION_2_1)
	{
		ErrorLog.Set("OpenGL 2.1 not supported!");
		return false;
	}

	if(MSAAPixelFormat == 0 && Samples > 0)
	{
		if(GLEW_ARB_multisample && WGLEW_ARB_pixel_format)
		{
			while(Samples > 0)
			{
				UINT NumFormats = 0;

				int PFAttribs[] =
				{
					WGL_DRAW_TO_WINDOW_ARB, GL_TRUE,
					WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
					WGL_DOUBLE_BUFFER_ARB, GL_TRUE,
					WGL_PIXEL_TYPE_ARB, WGL_TYPE_RGBA_ARB,
					WGL_COLOR_BITS_ARB, 32,
					WGL_DEPTH_BITS_ARB, 24,
					WGL_ACCELERATION_ARB, WGL_FULL_ACCELERATION_ARB,
					WGL_SAMPLE_BUFFERS_ARB, GL_TRUE,
					WGL_SAMPLES_ARB, Samples,
					0
				};

				if(wglChoosePixelFormatARB(hDC, PFAttribs, NULL, 1, &MSAAPixelFormat, &NumFormats) == TRUE && NumFormats > 0) break;

				Samples--;
			}

			wglDeleteContext(hGLRC);
			DestroyWindow(hWnd);
			UnregisterClass(WndClassEx.lpszClassName, hInstance);

			return Init(hInstance, Title, Width, Height, Samples);
		}
		else
		{
			Samples = 0;
		}
	}

	this->Samples = Samples;

	GetModuleDirectory();

	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &gl_max_texture_size);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &gl_max_texture_max_anisotropy_ext);
	}

	if(WGLEW_EXT_swap_control)
	{
		wglSwapIntervalEXT(0);
	}

	return OpenGLRenderer.Init();
}

void COpenGLView::Show(bool Maximized)
{
	RECT dRect, wRect, cRect;

	GetWindowRect(GetDesktopWindow(), &dRect);
	GetWindowRect(hWnd, &wRect);
	GetClientRect(hWnd, &cRect);

	wRect.right += Width - cRect.right;
	wRect.bottom += Height - cRect.bottom;
	wRect.right -= wRect.left;
	wRect.bottom -= wRect.top;
	wRect.left = dRect.right / 2 - wRect.right / 2;
	wRect.top = dRect.bottom / 2 - wRect.bottom / 2;

	MoveWindow(hWnd, wRect.left, wRect.top, wRect.right, wRect.bottom, FALSE);

	ShowWindow(hWnd, Maximized ? SW_SHOWMAXIMIZED : SW_SHOWNORMAL);
}

void COpenGLView::MessageLoop()
{
	MSG Msg;

	while(GetMessage(&Msg, NULL, 0, 0) > 0)
	{
		TranslateMessage(&Msg);
		DispatchMessage(&Msg);
	}
}

void COpenGLView::Destroy()
{
	if(GLEW_VERSION_2_1)
	{
		OpenGLRenderer.Destroy();
	}

	wglDeleteContext(hGLRC);
	DestroyWindow(hWnd);
}

void COpenGLView::OnKeyDown(UINT Key)
{
	switch(Key)
	{
		case VK_F1:
			OpenGLRenderer.Texturing = !OpenGLRenderer.Texturing;
			break;

		case VK_F2:
			OpenGLRenderer.NormalMapping = !OpenGLRenderer.NormalMapping;
			break;

		case VK_F3:
			OpenGLRenderer.CalculateShadows = !OpenGLRenderer.CalculateShadows;
			break;

		case VK_F4:
			OpenGLRenderer.DoShadowFiltering = !OpenGLRenderer.DoShadowFiltering;
			break;

		case VK_F5:
			OpenGLRenderer.CalculateSSAO = !OpenGLRenderer.CalculateSSAO;
			break;

		case VK_F6:
			OpenGLRenderer.ShowSSAO = !OpenGLRenderer.ShowSSAO;
			break;

		case VK_F7:
			OpenGLRenderer.BlurSSAO = !OpenGLRenderer.BlurSSAO;
			break;

		case VK_F8:
			OpenGLRenderer.FullSSAO = !OpenGLRenderer.FullSSAO;
			OpenGLRenderer.Resize(Width, Height);
			break;

		case VK_F9:
			OpenGLRenderer.CalculateAntialiasing = !OpenGLRenderer.CalculateAntialiasing;
			break;

		case VK_SPACE:
			OpenGLRenderer.Pause = !OpenGLRenderer.Pause;
			break;

		case 'g':
		case 'G':
			OpenGLRenderer.RenderGLUTObjects = !OpenGLRenderer.RenderGLUTObjects;
			break;
	}
}

void COpenGLView::OnMouseMove(int X, int Y)
{
	if(GetKeyState(VK_RBUTTON) & 0x80)
	{
		Camera.OnMouseMove(LastX - X, LastY - Y);

		LastX = X;
		LastY = Y;
	}
}

void COpenGLView::OnMouseWheel(short zDelta)
{
	Camera.OnMouseWheel(zDelta);
}

void COpenGLView::OnPaint()
{
	static DWORD LastFPSTime = GetTickCount(), LastFrameTime = LastFPSTime, FPS = 0;

	PAINTSTRUCT ps;

	HDC hDC = BeginPaint(hWnd, &ps);

	DWORD Time = GetTickCount();

	float FrameTime = (Time - LastFrameTime) * 0.001f;

	LastFrameTime = Time;

	if(Time - LastFPSTime > 1000)
	{
		CString Text = Title;

		if(OpenGLRenderer.Text[0] != 0)
		{
			Text.Append(" - " + OpenGLRenderer.Text);
		}

		Text.Append(" - %dx%d", Width, Height);
		Text.Append(", ATF %dx", gl_max_texture_max_anisotropy_ext);
		Text.Append(", MSAA %dx", Samples);
		Text.Append(", FPS: %d", FPS);
		Text.Append(" - %s", glGetString(GL_RENDERER));
		
		SetWindowText(hWnd, Text);

		LastFPSTime = Time;
		FPS = 0;
	}
	else
	{
		FPS++;
	}

	BYTE Keys = 0x00;

	if(GetKeyState('W') & 0x80) Keys |= 0x01;
	if(GetKeyState('S') & 0x80) Keys |= 0x02;
	if(GetKeyState('A') & 0x80) Keys |= 0x04;
	if(GetKeyState('D') & 0x80) Keys |= 0x08;
	if(GetKeyState('R') & 0x80) Keys |= 0x10;
	if(GetKeyState('F') & 0x80) Keys |= 0x20;

	if(GetKeyState(VK_SHIFT) & 0x80) Keys |= 0x40;
	if(GetKeyState(VK_CONTROL) & 0x80) Keys |= 0x80;

	if(Keys & 0x3F)
	{
		Camera.Move(Camera.OnKeys(Keys, FrameTime));
	}

	OpenGLRenderer.Render(FrameTime);

	SwapBuffers(hDC);

	EndPaint(hWnd, &ps);

	InvalidateRect(hWnd, NULL, FALSE);
}

void COpenGLView::OnRButtonDown(int X, int Y)
{
	LastX = X;
	LastY = Y;
}

void COpenGLView::OnSize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	OpenGLRenderer.Resize(Width, Height);
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView OpenGLView;

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uiMsg)
	{
		case WM_CLOSE:
			PostQuitMessage(0);
			break;

		case WM_MOUSEMOVE:
			OpenGLView.OnMouseMove(LOWORD(lParam), HIWORD(lParam));
			break;

		case 0x020A: // WM_MOUSWHEEL
			OpenGLView.OnMouseWheel(HIWORD(wParam));
			break;

		case WM_KEYDOWN:
			OpenGLView.OnKeyDown((UINT)wParam);
			break;

		case WM_PAINT:
			OpenGLView.OnPaint();
			break;

		case WM_RBUTTONDOWN:
			OpenGLView.OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_SIZE:
			OpenGLView.OnSize(LOWORD(lParam), HIWORD(lParam));
			break;

		default:
			return DefWindowProc(hWnd, uiMsg, wParam, lParam);
	}

	return 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow)
{
	char *AppName = "OpenGL 2.1 game engine";

	if(OpenGLView.Init(hInstance, AppName, 800, 600, 0))
	{
		OpenGLView.Show();
		OpenGLView.MessageLoop();
	}
	else
	{
		MessageBox(NULL, ErrorLog, AppName, MB_OK | MB_ICONERROR);
	}

	OpenGLView.Destroy();

	return 0;
}
