// ----------------------------------------------------------------------------------------------------------------------------
//
// Version 2.02
//
// ----------------------------------------------------------------------------------------------------------------------------

#include <windows.h>

#include "glmath.h"
#include "string.h"

#include <gl/glew.h> // http://glew.sourceforge.net/
#include <gl/wglew.h>

#include <FreeImage.h> // http://freeimage.sourceforge.net/

// ----------------------------------------------------------------------------------------------------------------------------

#pragma comment(lib, "opengl32.lib")
#pragma comment(lib, "glu32.lib")
#pragma comment(lib, "glew32.lib")
#pragma comment(lib, "FreeImage.lib")

// ----------------------------------------------------------------------------------------------------------------------------

extern CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

#define BUFFER_SIZE_INCREMENT 1048576

// ----------------------------------------------------------------------------------------------------------------------------

class CBuffer
{
private:
	BYTE *Buffer;
	int BufferSize, Position;

public:
	CBuffer();
	~CBuffer();

	void AddData(void *Data, int DataSize);
	void Empty();
	void *GetData();
	int GetDataSize();

private:
	void SetDefaults();
};

// ----------------------------------------------------------------------------------------------------------------------------

extern int gl_max_texture_size, gl_max_texture_max_anisotropy_ext;

// ----------------------------------------------------------------------------------------------------------------------------

class CTexture
{
protected:
	GLuint Texture;

public:
	CTexture();
	~CTexture();

	operator GLuint ();

	bool LoadTexture2D(char *FileName);
	bool LoadTextureCubeMap(char **FileNames);
	void Destroy();

protected:
	FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CShaderProgram
{
protected:
	GLuint VertexShader, FragmentShader, Program;

public:
	GLuint *UniformLocations, *AttribLocations;

public:
	CShaderProgram();
	~CShaderProgram();

	operator GLuint ();

	bool Load(char *VertexShaderFileName, char *FragmentShaderFileName);
	void Destroy();

protected:
	GLuint LoadShader(char *FileName, GLenum Type);
	void SetDefaults();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCamera
{
public:
	vec3 X, Y, Z, Position, Reference;

public:
	mat4x4 ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse, ViewProjectionMatrix, ViewProjectionMatrixInverse;

public:
	CCamera();
	~CCamera();

	void Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference = false);
	void Move(const vec3 &Movement);
	vec3 OnKeys(BYTE Keys, float FrameTime);
	void OnMouseMove(int dx, int dy);
	void OnMouseWheel(float zDelta);
	void SetPerspective(float fovy, float aspect, float n, float f);

private:
	void CalculateViewMatrix();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CIndexArrayElement
{
public:
	int v, c, tc, n;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CVertexArrayElement
{
public:
	vec3 Vertex;
	vec3 Color;
	vec2 TexCoord;
	vec3 Normal;
	vec3 Tangent;
	vec3 Bitangent;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CTriangle
{
public:
	vec3 A, B, C, M;
	vec3 AB, BC, CA;
	float LAB, LBC, LCA;
	vec3 N, NH, N1, N2, N3;
	float D, NdotNH, D1, D2, D3;
	vec3 HPNAB, HPNBC, HPNCA;
	float HPDAB, HPDBC, HPDCA;
	vec3 VPNAB, VPNBC, VPNCA;
	float VPDAB, VPDBC, VPDCA;

public:
	CTriangle();
	CTriangle(const vec3 &A, const vec3 &B, const vec3 &C);
	~CTriangle();

public:
	void Set(const vec3 &A, const vec3 &B, const vec3 &C);

public:
	bool Inside(const vec3 &Point);
	bool RayTriangleIntersectionTest(const vec3 &RayOrigin, const vec3 &RayDirection, float &MinDistance, vec3 &IntersectionPoint);
	bool GetHeightAbove(const vec3 &EyePosition, float &MinDistance, float &Height);
	bool GetHeightUnder(const vec3 &EyePosition, float EyeKneeDistance, float &MinDistance, float &Height);
	bool IntersectionTest(const vec3 &EyePositionA, const vec3 &EyePositionB, const vec3 &Direction, float EyeKneeDistance, float ClosestDistance, const vec3 &PN, float PD, float &MinDistance, vec3 &Compensation);
	bool DistanceTest(const vec3 &EyePositionB, float EyeKneeDistance, float ClosestDistance, float &MinDistance, vec3 &Compensation);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCollisionDetector
{
private:
	CTriangle *Triangles;
	int TrianglesCount;

private:
	float EyeHeight, EyeKneeDistance, ClosestDistance;

private:
	float EH, EHD2, EKD, EKDD2;

private:
	float FallSpeed;
	int CrouchState;

public:
	CCollisionDetector();
	~CCollisionDetector();

private:
	void SetDefaults();

public:
	void Init(vec3 *Vertices, int VerticesCount, float EyeHeight, float EyeKneeDistance, float ClosestDistance);
	void Destroy();

public:
	void Jump();
	void Crouch();

private:
	bool GetHeightAbove(const vec3 &EyePosition, float &MinDistance, float &Height);
	bool GetHeightUnder(const vec3 &EyePosition, float EyeKneeDistance, float &MinDistance, float &Height);
	bool IntersectionTest(const vec3 &EyePositionA, const vec3 &EyePositionB, const vec3 &Direction, float EyeKneeDistance, float ClosestDistance, const vec3 &PN, float PD, float &MinDistance, vec3 &Compensation);
	bool DistanceTest(const vec3 &EyePositionB, float EyeKneeDistance, float ClosestDistance, float &MinDistance, vec3 &Compensation);

public:
	void CheckHorizontalCollision(const vec3 &EyePosition, vec3 &Movement);
	void CheckVerticalCollision(const vec3 &EyePosition, float FrameTime, vec3 &Movement);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CObject
{
protected:
	char *FileName;

public:
	float sx, sy, sz, rx, ry, rz, tx, ty, tz;

protected:
	mat4x4 ModelMatrix;
	mat3x3 TangentMatrix, NormalMatrix;

protected:
	int VerticesCount, ColorsCount, TexCoordsCount, NormalsCount;

protected:
	int TrianglesCount;

protected:
	CTexture DiffuseMap, NormalMap;

protected:
	float SpecularIntensity, SpecularShininess;

protected:
	vec3 *Vertices;
	vec3 *Colors;
	vec2 *TexCoords;
	vec3 *Normals;

protected:
	CIndexArrayElement *IndexArray;

protected:
	CVertexArrayElement *VertexArray;
	GLuint VertexBufferObject;

protected:
	CTriangle *Triangles;

public:
	CObject();
	~CObject();

	bool Load(const char *FileName);
	void RenderIndexArray();
	/*void RenderVertexArray(int TangentAttribLocation = -1, int BitangentAttribLocation = -1);*/
	/*void RenderVertexBufferObject(int TangentAttribLocation = -1, int BitangentAttribLocation = -1);*/
	void RenderVertices();
	void RenderEdges(int TriangleIndex = -1);
	void RenderNormals();
	void Destroy();

	void CalculateModelMatrix();

	char* GetFileName();
	void SetModelMatrix(const mat4x4 &ModelMatrix);
	mat4x4& GetModelMatrix();
	mat3x3& GetTangentMatrix();
	mat3x3& GetNormalMatrix();
	GLuint GetDiffuseMap();
	GLuint GetNormalMap();
	float GetSpecularIntensity();
	float GetSpecularShininess();
	CTriangle* GetTriangles();
	int GetTrianglesCount();

	vec3 GetMiddle();

	bool RayTrianglesIntersectionTest(const vec3 &RayOrigin, const vec3 &RayDirection, float &MinDistance, vec3 &IntersectionPoint, int &TriangleIndex);

protected:
	void PrepareTriangles();
	/*void GenerateVertexArrayAndVertexBufferObject();*/
	void SetDefaults();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CScene
{
protected:
	CObject *Objects;
	int ObjectsCount;

public:
	CScene();
	~CScene();

	bool Load(const char *FileName);
	bool Save(const char *FileName);
	bool SaveBinary(const char *FileName);
	void RenderIndexArray(int ObjectIndex);
	void RenderVertices(int ObjectIndex);
	void RenderEdges(int ObjectIndex);
	void RenderNormals(int ObjectIndex);
	void Destroy();

	int GetObjectsCount();

	CTriangle* GetTriangles(int ObjectID);
	int GetTrianglesCount(int ObjectID);

	mat4x4& GetModelMatrix(int ObjectIndex);
	mat3x3& GetTangentMatrix(int ObjectIndex);
	mat3x3& GetNormalMatrix(int ObjectIndex);

	vec3 GetMiddle();

	void AddObject(CObject &Object);
	void DeleteObject(int ObjectIndex);

	bool RayTrianglesIntersectionTest(const vec3 &RayOrigin, const vec3 &RayDirection, float &MinDistance, vec3 &IntersectionPoint, int &TriangleIndex, int &ObjectIndex);

protected:
	void SetDefaults();
};

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLRenderer
{
protected:
	int LastX, LastY, LastClickedX, LastClickedY;

protected:
	int Width, Height;

protected:
	CCamera Camera;

protected:
	CCollisionDetector CollisionDetector;

protected:
	CShaderProgram Lighting;

protected:
	int SceneId;
	CString SceneFileName;
	CScene Scene;

protected:
	int ObjectId;
	CString ObjectFileName;
	CObject Object;

protected:
	int EditMode;

protected:
	vec3 IntersectionPlaneNormal, IntersectionPoint;
	float IntersectionPlaneD, IntersectionPointRoundFactor;
	int Index;
	bool IntersectionPointFound, ObjectFound;

protected:
	bool RenderVertices, RenderEdges, RenderNormals, ShowAxisGrid;

public:
	CString Text;

public:
	COpenGLRenderer();
	~COpenGLRenderer();

	bool Init();
	void Render(float FrameTime);
	void Resize(int Width, int Height);
	void Destroy();

	void InitCollisionDetector();

	void CheckCameraKeys(float FrameTime);

protected:
	void FindPrimitive();

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
};

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLView
{
protected:
	char *Title;
	int Width, Height, Samples;
	HWND hWnd;
	HGLRC hGLRC;

protected:
	COpenGLRenderer OpenGLRenderer;

public:
	COpenGLView();
	~COpenGLView();

	bool Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples);
	void Show(bool Maximized = false);
	void MessageLoop();
	void Destroy();

	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnPaint();
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
	void OnSize(int Width, int Height);
};

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow);
