// ----------------------------------------------------------------------------------------------------------------------------

#include <windows.h>

#ifndef WM_MOUSWHEEL
	#define WM_MOUSWHEEL 0x020A
#endif

#include "string.h"
#include "glmath.h"

#include <FreeImage.h> // http://freeimage.sourceforge.net/

#pragma comment(lib, "FreeImage.lib")
#pragma comment(lib, "winmm.lib")

#pragma warning(disable : 4996)

// ----------------------------------------------------------------------------------------------------------------------------

class CTexture
{
private:
	FIBITMAP *DIB;
	BYTE *Bits;

private:
	int Width, Height, Pitch, BPP;

public:
	CTexture();
	~CTexture();

private:
	void SetDefaults();

public:
	bool LoadTexture(char *TextureFileName);
	void GetColorNearest(float s, float t, float *r, float *g, float *b);
	void GetColorBilinear(float s, float t, float *r, float *g, float *b);
	void Destroy();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCamera
{
public:
	vec3 X, Y, Z, Position, Reference;

public:
	mat4x4 ViewMatrix, ProjectionMatrix, ViewProjectionMatrix;

public:
	CCamera();
	~CCamera();

public:
	void Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference = false);
	void Move(const vec3 &Movement);
	vec3 OnKeys(BYTE Keys, float FrameTime);
	void OnMouseMove(int dx, int dy);
	void OnMouseWheel(float zDelta);
	void SetPerspective(float fovy, float aspect, float n, float f);

private:
	void CalculateViewMatrix();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CVertex
{
public:
	vec3 Position;
	vec3 Color;
	vec2 TexCoord;
	vec3 Normal;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CObject
{
public:
	CTexture Texture;

public:
	CVertex *Vertices;
	int VerticesCount;

public:
	vec3 Min, Max;

public:
	CObject();
	~CObject();

private:
	void SetDefaults();

public:
	bool Load(char *Directory, char *ObjFileName);
	void Translate(const vec3 &Translation);
	void Scale(float ScaleFactor);
	void Rotate(float Angle, const vec3 &Axis);
	void Destroy();

private:
	bool ReadSource(char *Directory, char *FileName, char **Source, long &Length);
	bool ParseMtl(char *Directory, char *MtlFileName);
	void GetMinMax();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CLight
{
public:
	vec3 Position;
	vec3 Ambient;
	vec3 Diffuse;
	float ConstantAttenuation;
	float LinearAttenuation;
	float QuadraticAttenuation;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CFragment
{
public:
	float x, y, z, w, r, g, b, s, t;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CEdge
{
public:
	CFragment *Fragment1, *Fragment2;

public:
	float dx, dy, dz, dw, dr, dg, db, ds, dt;

public:
	void Set(CFragment *Fragment1, CFragment *Fragment2);

private:
    void CalculateDiffs();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CSpan
{
public:
	CFragment *Fragment1, *Fragment2;

public:
	float dx, dy, dz, dw, dr, dg, db, ds, dt;

public:
	void Set(CFragment *Fragment1, CFragment *Fragment2);

private:
    void CalculateDiffs();
};

// ----------------------------------------------------------------------------------------------------------------------------

#define NONE 0x00

#define CULL_FACE_FRONT 0x01
#define CULL_FACE_BACK 0x02

#define ANTI_ALIASING_2X2 0x03
#define ANTI_ALIASING_3X3 0x04
#define ANTI_ALIASING_4X4 0x05

// ----------------------------------------------------------------------------------------------------------------------------

class CSoftwareGL
{
private:
	int Width, Height;

private:
	BYTE *AntiAliasingColorBuffer;
	int AntiAliasingColorBufferWidth, AntiAliasingColorBufferHeight;

private:
	BYTE *StandardColorBuffer;
	int StandardColorBufferWidth, StandardColorBufferHeight;
	BITMAPINFO StandardColorBufferInfo;

private:
	BYTE *ColorBuffer;
	int ColorBufferWidth, ColorBufferHeight;

private:
	USHORT *DepthBuffer;
	int DepthBufferWidth, DepthBufferHeight;

private:
	int AntiAliasing;
	mat4x4 ModelViewProjectionMatrix;
	CLight *Light;
	int CullFace;
	bool BilinearTextureFiltering;
	CTexture *Texture;

public:
	CSoftwareGL();
	~CSoftwareGL();

public:
	int GetAntiAliasing();
	void SetAntiAliasing(int AntiAliasing);
	int GetCullFace();
	void SetCullFace(int CullFace);
	bool GetBilinearTextureFiltering();
	void SetBilinearTextureFiltering(bool BilinearTextureFiltering);

public:
	void Clear();
	void LoadModelViewProjectionMatrix(const mat4x4 &ModelViewProjectionMatrix);
	void BindLight(CLight *Light);
	void BindTexture(CTexture *Texture);
	void DrawTriangles(CVertex *Vertices, int FirstIndex, int Count);

private:
	void ClipTriangle(CFragment *Fragments, int ClipPlane = 1);
	void RasterizeTriangle(CFragment *Fragments);
	void DrawSpansBetweenEdges(CEdge *Edge1, CEdge *Edge2);
	void DrawSpan(CSpan *Span, int y);
	void BlitAntiAliasingColorBuffer2x2();
	void BlitAntiAliasingColorBuffer3x3();
	void BlitAntiAliasingColorBuffer4x4();

public:
	void Viewport(int Width, int Height);
	void SwapBuffers(HDC hDC);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CSoftwareGLRenderer : public CSoftwareGL
{
private:
	int LastX, LastY, LastClickedX, LastClickedY;

private:
	CCamera Camera;

private:
	CTexture Texture;

private:
	CVertex *Vertices;

private:
	CObject Object[3];

private:
	int RenderObject;

private:
	bool Lighting, Texturing;

private:
	CLight Light;

public:
	CSoftwareGLRenderer();
	~CSoftwareGLRenderer();

public:
	bool Init();
	void Render(float FrameTime);
	void Resize(int Width, int Height);
	void Destroy();

public:
	void CheckCameraKeys(float FrameTime);

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CSoftwareGLView
{
private:
	char *Title;
	int Width, Height;
	HWND hWnd;
	HDC hDC;

private:
	CSoftwareGLRenderer SoftwareGLRenderer;

public:
	CSoftwareGLView();
	~CSoftwareGLView();

public:
	bool Create(HINSTANCE hInstance, char *Title, int Width, int Height);
	void Show(bool Maximized = false);
	void MsgLoop();
	void Destroy();

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnPaint();
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
	void OnSize(int Width, int Height);
};

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow);

// ----------------------------------------------------------------------------------------------------------------------------
