// ----------------------------------------------------------------------------------------------------------------------------
//
// Version 2.02
//
// ----------------------------------------------------------------------------------------------------------------------------

#include <windows.h>

#include "glmath.h"
#include "string.h"

#include <gl/glew.h> // http://glew.sourceforge.net/
#include <gl/wglew.h>

#include <FreeImage.h> // http://freeimage.sourceforge.net/

// ----------------------------------------------------------------------------------------------------------------------------

#pragma comment(lib, "opengl32.lib")
#pragma comment(lib, "glu32.lib")
#pragma comment(lib, "glew32.lib")
#pragma comment(lib, "FreeImage.lib")

// ----------------------------------------------------------------------------------------------------------------------------

extern CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

#define BUFFER_SIZE_INCREMENT 1048576

// ----------------------------------------------------------------------------------------------------------------------------

class CBuffer
{
private:
	BYTE *Buffer;
	int BufferSize, Position;

public:
	CBuffer();
	~CBuffer();

	void AddData(void *Data, int DataSize);
	void Empty();
	void *GetData();
	int GetDataSize();

private:
	void SetDefaults();
};

// ----------------------------------------------------------------------------------------------------------------------------

extern int gl_max_texture_size, gl_max_texture_max_anisotropy_ext;

// ----------------------------------------------------------------------------------------------------------------------------

class CTexture
{
protected:
	GLuint Texture;

public:
	CTexture();
	~CTexture();

	operator GLuint ();

	bool LoadTexture2D(char *FileName);
	bool LoadTextureCubeMap(char **FileNames);
	void Destroy();

protected:
	FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CShaderProgram
{
protected:
	GLuint VertexShader, FragmentShader, Program;

public:
	GLuint *UniformLocations, *AttribLocations;

public:
	CShaderProgram();
	~CShaderProgram();

	operator GLuint ();

	bool Load(char *VertexShaderFileName, char *FragmentShaderFileName);
	void Destroy();

protected:
	GLuint LoadShader(char *FileName, GLenum Type);
	void SetDefaults();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCamera
{
protected:
	mat4x4 *ViewMatrix, *ViewMatrixInverse;

public:
	vec3 X, Y, Z, Position, Reference;

	CCamera();
	~CCamera();

	void Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference = false);
	void Move(const vec3 &Movement);
	vec3 OnKeys(BYTE Keys, float FrameTime);
	void OnMouseMove(int dx, int dy);
	void OnMouseWheel(float zDelta);
	void SetViewMatrixPointer(float *ViewMatrix, float *ViewMatrixInverse = NULL);

private:
	void CalculateViewMatrix();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CTriangle
{
public:
	float D, D1, D2, D3, lab, lbc, lca;
	vec3 a, b, c, ab, bc, ca, Color, N, N1, N2, N3;

public:
	CTriangle();
	CTriangle(const vec3 &a, const vec3 &b, const vec3 &c, const vec3 &Color);

	bool Inside(float x, float y, float z);
	bool Inside(const vec3 &Point);
	bool Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance, float &Distance, vec3 &Point);
	bool Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance, float &Distance);
	bool Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance);
};

// ----------------------------------------------------------------------------------------------------------------------------

class RTData
{
public:
	float Distance, TestDistance;
	vec3 Color, Point, TestPoint;
	CTriangle *Triangle;

public:
	RTData();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CVoxel
{
public:
	CTriangle **Triangles;
	int TrianglesCount;

protected:
	int MaxTrianglesCount;
	float Size;
	vec3 Min, Max, MinE, MaxE;

public:
	CVoxel();
	~CVoxel();

	void Add(CTriangle *Triangle);
	void Delete();
	bool Inside(const vec3 &Point);
	bool Intersect(CTriangle *Triangle);
	bool IntersectEdgesX(CTriangle *Triangle, float x, float y1, float y2, float z1, float z2);
	bool IntersectEdgesY(CTriangle *Triangle, float y, float x1, float x2, float z1, float z2);
	bool IntersectEdgesZ(CTriangle *Triangle, float z, float x1, float x2, float y1, float y2);
	bool IntersectFacesX(CTriangle *Triangle, float D1, float D2);
	bool IntersectFacesY(CTriangle *Triangle, float D1, float D2);
	bool IntersectFacesZ(CTriangle *Triangle, float D1, float D2);
	void Render();
	void Set(const vec3 &Min, float Size);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CUniformGrid
{
protected:
	vec3 Min, Max;

protected:
	int X, Y, Z, Xm1, Ym1, Zm1, XY, XYZ;
	float VoxelSize;
	CVoxel *Voxels;

public:
	CUniformGrid();
	~CUniformGrid();

public:
	void Delete();
	void Generate(CTriangle *Triangles, int TrianglesCount, float VoxelSize = 1.0f);
	void RenderGrid();
	vec3 Traverse(const vec3 &Voxel, const vec3 &Origin, const vec3 &Ray);

	float VoxelToWorldX(float x);
	float VoxelToWorldY(float y);
	float VoxelToWorldZ(float z);
	vec3 VoxelToWorld(const vec3 &Voxel);
	int WorldToVoxelX(float x);
	int WorldToVoxelY(float y);
	int WorldToVoxelZ(float z);
	vec3 WorldToVoxel(const vec3 &World);
};

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLRenderer
{
protected:
	int Width, Height;
	mat3x3 NormalMatrix;
	mat4x4 ModelMatrix, ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse;

protected:
	CTriangle *Triangles;
	int TrianglesCount;

private:
	CUniformGrid UniformGrid;

public:
	vec3 Start, End;

public:
	CString Text;

public:
	COpenGLRenderer();
	~COpenGLRenderer();

	bool Init();
	void Render(float FrameTime);
	void Resize(int Width, int Height);
	void Destroy();

	void Check(int x, int y);
};

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLView
{
protected:
	char *Title;
	int Width, Height, Samples;
	HWND hWnd;
	HGLRC hGLRC;

protected:
	int LastX, LastY;

public:
	COpenGLView();
	~COpenGLView();

	bool Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples);
	void Show(bool Maximized = false);
	void MessageLoop();
	void Destroy();

	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnPaint();
	void OnRButtonDown(int X, int Y);
	void OnSize(int Width, int Height);
};

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow);
